/* ------------------------------------------------------------------------
Bill Buckels
Aug 2014

Default

Reads a 16 color, 256 color, or 24 bit BMP and writes a set of full-screen
binary DHGR files in the standard A2FC single file format, with a raw
Auxiliary DHGR Memory "DUMP" of 8192 bytes, followed by a raw Main DHGR Memory
"DUMP of 8192 bytes, totalling 16384 bytes. These are stored in ProDOS as
Binary FileType $06 with an Auxiliary Type of either $2000 or $4000, which is
the load address of the DHGR screen.

Quick Commands:

b2d input.bmp photo
b2d input.bmp art
b2d input.bmp both
b2d input.bmp sprite
b2d input.bmp bin
b2d help

Input File

Any 16 color, 256 color, or 24-bit BMP Version 3 file of 140 x 192 or 280 x
192 resolutions will result in full-screen output in one of 2 formats:

1. A2FC or AUX,BIN File Pairs
2. "Sprite" (image fragment) DHR files.

Any BMP in this range can also be double-scaled when optional "Sprite" output
is selected.

When scale is not set the maximum BMP input resolution for "Sprites" is 140 x
192, and when scale is set the maximum is 280 x 192.

140 x 192 - sprite or verbatim scale (scaled gif conversions, etc)
280 x 192 - sprite or double xscale (screen grabs, etc.)

In addition to the sprite range, the following classic BMP sizes are supported
for full-screen conversion:

320 x 200 - legacy graphics
560 x 384 - screen grabs or cropped to scale input (pixels to pixel)
640 x 400 - legacy graphics
640 x 480 - photos and legacy graphics, wikipedia images, wallpaper, etc.

Alternate Default Output

Alternate output of a split version of the A2FC format is optionally
available. Sometimes called AUX,BIN file pairs, these are easier to load in an
AppleSoft BASIC program. They are also stored in ProDOS as Binary FileType $06
with an Auxiliary Type of either $2000 or $4000 and are 8192 bytes each.

These are indistinguishable from normal HGR binary files which are also the
same type and length of 8192 bytes so a loader must be aware of the specific
files, or must use a naming convention like AUX,BIN extensions respectively
and check for the presense of both before attempting to load them as DHGR
"DUMP" files.

Optional Image Fragment Output

Sprite (image fragment) format Output is an option.

Sprite Output and normal DHGR full-screen output are mutually exclusive.

About Sprites

The Sprites produced by this utility are in XPACK's DHR format. but XPACK only
produces Full Screen DHGR images.

This utility produces both Full Screen images or Image Fragments (Sprites).

The DHR (Double Hi-Res Raster) Image Format

Some utilities call their "DUMP" file formats DHR files. bmp2dhr (this
utility) follows a more rigid naming approach and calls its dump files by the
traditional naming loosely supported by Beagle Graphics (AUX,BIN) and others.
The originator of the naming convention A2FC (Apple II Full-Color) is unknown
to me, but following a naming convention while not necessary is helpful.
Calling something something else that conflicts with a naming convention and
ignores other established naming conventions is not helpful. Ignoring
extensions altogther is less than not helpful.

The image fragments produced by bmp2dhr have an extension of DHR.Like A2FC and
AUX,BIN file pairs, they are stored as ProDOS FileType $06 with an Auxiliary
Type of $2000 or $4000 by default. However if a programmer wanted to load
these according to a specific position on the DHGR display it would be
possible to give the starting scanline and starting offset to the desired
position of the first raster, and store that as the Auxiliary Type instead:

1. The program would read the 2 byte header and perform a file integrity check
to ensure that the file size was as expected.

2. Part of the verification would also be to determine if the Auxiliary Type
fell within the DHGR visible screen boundaries and if the file itself would
fit.

3. Having satisfied this requirement the image fragment could be positioned at
that point by the program.

Doing so would save disk-space and load time when constructing a pre-planned
screen in a DHGR program, since full-screens are large by comparison to
creating full-screens from fragments.

The DHR is a raster based image with scanlines of raw DHGR data alternating
between auxiliary and main memory. Therefore a simple BASIC program cannot
easily load these since the DHGR screen is interleaved the same way that the
HGR screen is interleaved and not linear. Bank switching between auxiliary and
main memory banks 0 (main board) and 1 (language card) is also not easy in a
BASIC program.

For a full-screen DHR, there are 192 pairs of rasters of 40 bytes of auxiliary
memory data followed by 40 bytes of main memory data.

The full screen DHR loads raster by raster and displays as quickly as a
buffered read can display on the Apple II. At 15360 bytes it provides a modest
disk space saving over the 16384 bytes of the A2FC or AUX,BIN equivalent.

A caveat for any file in DHR raster format is the 4 byte / 7 pixel pattern of
the DHGR display. The descriptor in the 2 byte header is given in byte width
rather than pixel width. Image fragments in DHGR must necessarily be aligned
on 4 byte boundaries to display properly. This utility pads DHR formats as
required in an optional background color if desired.

By comparison, HGR image fragments are aligned on 2 byte boundaries for proper
display but they are still somewhat recognizable if not aligned properly.If
DHGR image fragments are not aligned on 4 byte boundaries they are a mess.

This program has many more options. The source code (this file) and the
documentation can be reviewed for additional information.

Licence      : You may use this program for whatever you wish as long
			   as you agree that Bill Buckels has no warranty or
			   liability obligations whatsoever from said use.
------------------------------------------------------------------------ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <math.h>

#include "tomthumb.h"

/* ***************************************************************** */
/* ========================== defines ============================== */
/* ***************************************************************** */

#define SUCCESS 0
#define INVALID -1
#define RETRY 911
#define BIN_OUTPUT 1
#define SPRITE_OUTPUT 2

/* constants for the biCompression field in a BMP header */
#define BI_RGB      0L
#define BI_RLE8     1L
#define BI_RLE4     2L


/* DHGR, DLGR, LGR colors - LGR color order */
#define LOBLACK   	0
#define LORED     	1
#define LODKBLUE 	2
#define LOPURPLE  	3
#define LODKGREEN	4
#define LOGRAY    	5
#define LOMEDBLUE	6
#define LOLTBLUE 	7
#define LOBROWN   	8
#define LOORANGE  	9
#define LOGREY    	10
#define LOPINK    	11
#define LOLTGREEN	12
#define LOYELLOW  	13
#define LOAQUA    	14
#define LOWHITE   	15

#define RED    0
#define GREEN  1
#define BLUE   2

#define FLOYDSTEINBERG 1
#define JARVIS 2
#define STUCKI 3
#define ATKINSON 4
#define BURKES 5
#define SIERRA 6
#define SIERRATWO 7
#define SIERRALITE 8
#define BUCKELS 9
#define CUSTOM 10

#define ASCIIZ	0
#define CRETURN 13
#define LFEED	10

/* ***************************************************************** */
/* ========================== typedefs ============================= */
/* ***************************************************************** */

typedef unsigned char uchar;
typedef unsigned short ushort;
typedef unsigned long ulong;
typedef short sshort;

/* Bitmap Header structures */
#ifdef MINGW
typedef struct __attribute__((__packed__)) tagBITMAPINFOHEADER
#else
typedef struct tagBITMAPINFOHEADER
#endif
{
    ulong   biSize;
    ulong   biWidth;
    ulong   biHeight;
    ushort  biPlanes;
    ushort  biBitCount;
    ulong   biCompression;
    ulong   biSizeImage;
    ulong   biXPelsPerMeter;
    ulong   biYPelsPerMeter;
    ulong   biClrUsed;
    ulong   biClrImportant;
} BITMAPINFOHEADER;

#ifdef MINGW
typedef struct __attribute__((__packed__)) tagBITMAPFILEHEADER
#else
typedef struct tagBITMAPFILEHEADER
#endif
{
    uchar   bfType[2];
    ulong   bfSize;
    ushort  bfReserved1;
    ushort  bfReserved2;
    ulong   bfOffBits;
} BITMAPFILEHEADER;

#ifdef MINGW
typedef struct __attribute__((__packed__)) tagBMPHEADER
#else
typedef struct tagBMPHEADER
#endif
{
BITMAPFILEHEADER bfi;
BITMAPINFOHEADER bmi;
} BMPHEADER;


#ifdef MINGW
typedef struct __attribute__((__packed__)) tagRGBQUAD
#else
typedef struct tagRGBQUAD
#endif
{
    unsigned char    rgbBlue;
    unsigned char    rgbGreen;
    unsigned char    rgbRed;
    unsigned char    rgbReserved;
} RGBQUAD;


/* ***************************************************************** */
/* ========================== globals ============================== */
/* ***************************************************************** */

int mono = 0;

sshort GetUserTextFile(void);
int dhrgetpixel(int x,int y);

/* static structures for processing */
BITMAPFILEHEADER bfi;
BITMAPINFOHEADER bmi;
BMPHEADER mybmp,maskbmp;
RGBQUAD   sbmp[256], maskpalette[256]; /* super vga - applewin */

/* overlay file for screen titling and framing and "cleansing" */
FILE *fpmask = NULL;
unsigned char remap[256];

char bmpfile[256], dibfile[256], scaledfile[256], previewfile[256], reformatfile[256], maskfile[256];
char spritefile[256], mainfile[256],auxfile[256],a2fcfile[256],usertextfile[256],vbmpfile[256],fname[256];
int backgroundcolor = 0, quietmode = 1, diffuse = 0, merge = 0, scale = 0, applesoft = 0, outputtype = BIN_OUTPUT;
int reformat = 0, debug = 0;
int preview = 0, vbmp = 0;
int overlay = 0, maskpixel=0, overcolor=0, clearcolor=5;

/* 2 x 2 general purpose cross-hatching */
int xmatrix = 0, ymatrix = 0, threshold = 0;

ushort bmpwidth = 0, bmpheight = 0, spritewidth = 0;

/* classic size alternate windowed output */
/* applies to 320 x 200, 640 x 400, and 640 x 480 only */
/* for 320 x 200 the window is 280 x 192 */
/* for 640 x the window is 560 x 384 */
/* output is always 280 x 192 */
/* for 640 output is merged 2 x 2 */
sshort justify = 0, jxoffset = -1, jyoffset = -1;


/* the line buffers in this program are designed to hold a 320 pixel scanline...
   for reusability in other modules and for other Apple II modes
   including SHR 320 */
/* for DHGR monochrome they are designed to hold a scanline 560 pixels wide */
uchar bmpscanline[1920],
      bmpscanline2[1920],
      dibscanline1[1920],
      dibscanline2[1920],
      dibscanline3[1920],
      dibscanline4[1920],
      previewline[1920],
      maskline[140];


/* riemersa dithering */
#define WIDTH   140
#define HEIGHT  192

#define BLOCKSIZE 16384 // read in chunks of 16 kBytes

enum {
  NONE,
  UP,
  LEFT,
  DOWN,
  RIGHT,
};

/* variables needed for the Riemersma dither algorithm */
static int cur_x=0, cur_y=0;
static int img_width=0, img_height=0;
static unsigned char *img_ptr;

#define SIZE 16                 /* queue size: number of pixels remembered */
#define MAX  16                 /* relative weight of youngest pixel in the
                                 * queue, versus the oldest pixel */

static int weights[SIZE];       /* weights for the errors of recent pixels */

static void init_weights(int a[],int size,int max)
{
  double m = exp(log(max)/(size-1));
  double v;
  int i;

  for (i=0, v=1.0; i<size; i++) {
    a[i]=(int)(v+0.5);  /* store rounded value */
    v*=m;               /* next value */
  } /*for */
}

static void dither_pixel(unsigned char *pixel)
{
static int error[SIZE]; /* queue with error values of recent pixels */
  int i,pvalue,err;

  for (i=0,err=0L; i<SIZE; i++)
    err+=error[i]*weights[i];
  pvalue=*pixel + err/MAX;
  pvalue = (pvalue>=128) ? 255 : 0;
  memmove(error,error+1,(SIZE-1)*sizeof error[0]);    /* shift queue */
  error[SIZE-1] = *pixel - pvalue;
  *pixel=(unsigned char)pvalue;
}

static void move(int direction)
{
  /* dither the current pixel */
  if (cur_x>=0 && cur_x<img_width && cur_y>=0 && cur_y<img_height)
    dither_pixel(img_ptr);

  /* move to the next pixel */
  switch (direction) {
  case LEFT:
    cur_x--;
    img_ptr--;
    break;
  case RIGHT:
    cur_x++;
    img_ptr++;
    break;
  case UP:
    cur_y--;
    img_ptr-=img_width;
    break;
  case DOWN:
    cur_y++;
    img_ptr+=img_width;
    break;
  } /* switch */
}

void hilbert_level(int level,int direction)
{
  if (level==1) {
    switch (direction) {
    case LEFT:
      move(RIGHT);
      move(DOWN);
      move(LEFT);
      break;
    case RIGHT:
      move(LEFT);
      move(UP);
      move(RIGHT);
      break;
    case UP:
      move(DOWN);
      move(RIGHT);
      move(UP);
      break;
    case DOWN:
      move(UP);
      move(LEFT);
      move(DOWN);
      break;
    } /* switch */
  } else {
    switch (direction) {
    case LEFT:
      hilbert_level(level-1,UP);
      move(RIGHT);
      hilbert_level(level-1,LEFT);
      move(DOWN);
      hilbert_level(level-1,LEFT);
      move(LEFT);
      hilbert_level(level-1,DOWN);
      break;
    case RIGHT:
      hilbert_level(level-1,DOWN);
      move(LEFT);
      hilbert_level(level-1,RIGHT);
      move(UP);
      hilbert_level(level-1,RIGHT);
      move(RIGHT);
      hilbert_level(level-1,UP);
      break;
    case UP:
      hilbert_level(level-1,LEFT);
      move(DOWN);
      hilbert_level(level-1,UP);
      move(RIGHT);
      hilbert_level(level-1,UP);
      move(UP);
      hilbert_level(level-1,RIGHT);
      break;
    case DOWN:
      hilbert_level(level-1,RIGHT);
      move(UP);
      hilbert_level(level-1,DOWN);
      move(LEFT);
      hilbert_level(level-1,DOWN);
      move(DOWN);
      hilbert_level(level-1,LEFT);
      break;
    } /* switch */
  } /* if */
}

int mylog2(int value)
{
  int result=0;
  while (value>1) {
    value >>= 1;
    result++;
  } /*while */
  return result;
}


int mymax(int a, int b)
{
	if (a > b) return a;
	return b;
}

void Riemersma(void *image,int width,int height)
{
  int level,size;

  /* determine the required order of the Hilbert curve */
  size=mymax(width,height);
  level=mylog2(size);
  if ((1L << level) < size)
    level++;

  init_weights(weights,SIZE,MAX);
  img_ptr=image;
  img_width=width;
  img_height=height;
  cur_x=0;
  cur_y=0;
  if (level>0)
    hilbert_level(level,UP);
  move(NONE);
}




/* Floyd-Steinberg Etc. Dithering */
uchar dither = 0, errorsum = 0, serpentine = 0;
/* color channel buffers for Floyd-Steinberg Etc. Dithering */
/* note that these are arrays of short integers (2 bytes * 320) */
/* current line contains current scanline values
   + (plus) seed values (error values) from previous line

   these error values are the linear distance in a scale of 0-255
   between the real color and the nearest palette color.

   each of the r,g,b color channels is separate.

   a psychovisual palette match is done with the real color
   to get the near color from the palette

   the dither error is then applied to neighboring pixels
   using the dithering filter's distribution pattern

   later on when the color channels are finally reconstructed
   a psychovisual palette match is done again to get the dither
   color.

   if preview is turned-on the output can be viewed in
   a BMP viewer.

*/
sshort redDither[640],greenDither[640],blueDither[640];
/* seed values from previous line */
sshort redSeed[640],greenSeed[640],blueSeed[640];
sshort redSeed2[640],greenSeed2[640],blueSeed2[640];
sshort *colorptr, *seedptr, *seed2ptr, color_error;

int colorbleed = 100;

char *dithertext[] = {
	"Floyd-Steinberg",
	"Jarvis",
	"Stucki",
	"Atkinson",
	"Burkes",
	"Sierra",
	"Sierra Two",
	"Sierra Lite",
	"Buckels",
	"Custom"};


/* built-in palette options */
/* based on DHGR, DLGR, LGR colors */
/* LGR color order */
uchar kegs32colors[16][3] = {
	0,0,0,		 /* black    */
	221,0,51,	 /* red      */
	0,0,153,	 /* dk blue  */
	221,0,221,	 /* purple   */
	0,119,0,	 /* dk green */
	85,85,85,	 /* gray     */
	34,34,255,	 /* med blue */
	102,170,255, /* lt blue  */
	136,85,34,	 /* brown    */
	255,102,0,	 /* orange   */
	170,170,170, /* grey     */
	255,153,136, /* pink     */
	0,221,0,	 /* lt green */
	255,255,0,	 /* yellow   */
	0,255,153,	 /* aqua     */
	255,255,255};/* white    */

uchar ciderpresscolors[16][3] = {
	0,0,0,		 /* black    */
	221,0,51,	 /* red      */
	0,0,153,	 /* dk blue  */
	221,34,221,	 /* purple   */
	0,119,34,	 /* dk green */
	85,85,85,	 /* gray     */
	34,34,255,	 /* med blue */
	102,170,255, /* lt blue  */
	136,85,0,	 /* brown    */
	255,102,0,	 /* orange   */
	170,170,170, /* grey     */
	255,153,136, /* pink     */
	17,221,0,	 /* lt green */
	255,255,0,	 /* yellow   */
	68,255,153,	 /* aqua     */
	255,255,255};/* white    */

uchar awinoldcolors[16][3] = {
	0,0,0,       /* black    */
	208,0,48,	 /* red      */
	0,0,128,	 /* dk blue  */
	255,0,255,	 /* purple   */
	0,128,0,	 /* dk green */
	128,128,128, /* gray     */
	0,0,255,	 /* med blue */
	96,160,255,	 /* lt blue  */
	128,80,0,	 /* brown    */
    255,128,0,	 /* orange   */
    192,192,192, /* grey     */
    255,144,128, /* pink     */
    0,255,0,	 /* lt green */
    255,255,0,	 /* yellow   */
    64,255,144,	 /* aqua     */
    255,255,255};/* white    */

uchar awinnewcolors[16][3] = {
	0,0,0,       /* black    */
	157,9,102,   /* red      */
	42,42,229,   /* dk blue  */
	199,52,255,  /* purple   */
	0,118,26,    /* dk green */
	128,128,128, /* gray     */
	13,161,255,  /* med blue */
	170,170,255, /* lt blue  */
	85,85,0,     /* brown    */
    242,94,0,    /* orange   */
    192,192,192, /* grey     */
    255,137,229, /* pink     */
    56,203,0,    /* lt green */
    213,213,26,  /* yellow   */
    98,246,153,  /* aqua     */
    255,255,255};/* white    */

/* http://en.wikipedia.org/wiki/List_of_8-bit_computer_hardware_palettes */
uchar wikipedia[16][3] = {
	0,0,0,		 /* black */
	114,38,64,   /* red */
	64,51,127,   /* dk blue */
	228,52,254,  /* purple */
	14,89,64,    /* dk green */
	128,128,128, /* gray */
	27,154,254,  /* med blue */
	191,179,255, /* lt blue */
	64,76,0,     /* brown */
	228,101,1,   /* orange */
	128,128,128, /* grey */
	241,166,191, /* pink */
	27,203,1,    /* lt green */
	191,204,128, /* yellow */
	141,217,191, /* aqua */
	255,255,255};/* white */

/* http://wsxyz.net/tohgr.html */
/* Sheldon Simm's palette from todhr */
uchar grpal[16][3] = {
	0x00,0x00,0x00, /* black */
	0xad,0x18,0x28, /* red */
	0x55,0x1b,0xe1, /* dk blue */
	0xe8,0x2c,0xf8, /* purple */
	0x01,0x73,0x63, /* dk green */
	0x7e,0x82,0x7f, /* gray */
	0x34,0x85,0xfc, /* med blue */
	0xd1,0x95,0xff, /* lt blue */
	0x33,0x6f,0x00, /* brown */
	0xd0,0x81,0x01, /* orange */
	0x7f,0x7e,0x77, /* grey */
	0xfe,0x93,0xa3, /* pink */
	0x1d,0xd6,0x09, /* lt green */
	0xae,0xea,0x22, /* yellow */
	0x5b,0xeb,0xd9, /* aqua */
	0xff,0xff,0xff};/* white */
/*
 0,  0,  0,	  black
173, 24, 40,  red
 85, 27,225,  dk blue
232, 44,248,  purple
  1,115, 99,  dk green
126,130,127,  gray
 52,133,252,  med blue
209,149,255,  lt blue
 51,111,  0,  brown
208,129,  1,  orange
127,126,119,  grey
254,147,163,  pink
 29,214,  9,  lt green
174,234, 34,  yellow
 91,235,217,  aqua
255,255,255}; white


*/

/* ------------------------------------------------------------------------------ */
/* skip this if you think that this program has been clean up to now.             */
/* this is more like an egg I laid than an easter egg.                            */
/* some legacy stuff - undocumented - unhacked skeletons in my closet             */
/* from the AppleX Aztec C65 distribution - not everything always goes as planned */
/* ------------------------------------------------------------------------------ */

/* "designer" palettes - VGA style mapping colors from BMPA2FC */
/* these are in the lo-res color order, same as above */
/* in BMPA2FC these were in IBM-PC default BIOS order */

/* Serious doubt! */
/* at this point, I wonder how well BMPA2FC mapped colors from BMP's */
/* not very well I think - but it's an old MS-DOS utility and no point
   in worrying about it now - new users who find this can play with it */

/* this first array was my first attempt at establishing something that looked like
   24 bit Lo-Res Colors - I am not even sure where I got the colors
   they looked like what I wanted at the time before I knew a little more

   the rest of the palettes in here were just pasted forward from ClipShop
   but with a couple of minor changes to accomodate the move to Windows Paint for Xp
   and the 16 color BMP's which have been stable ever since.

   why the default palette changed between XP and Windows 3.1 I have no clue.

   */

/* I am not documenting these because I don't want to propagate misinformation
   more than usual, but if someone happens to use my canvas scheme from
   my old utilities or whatever, having these here will at least save them
   from ending-up with work that needs to be redone if they expected remapping */

/* goes with the canvas bmp that I put out there with AppleX */
uchar rgbCanvasArray[16][3] = {
0  , 0  , 0  ,
208, 0  , 48 ,
0  , 0  , 128,
255, 0  , 255,
0  , 128, 0  ,
128, 128, 128,
0  , 0  , 255,
96 , 160, 255,
128, 80 , 0  ,
255, 128, 0  ,
192, 192, 192,
255, 144, 128,
0  , 255, 0  ,
255, 255, 0  ,
64 , 255, 144,
255, 255, 255};

/* this might work with old Win16 16 color BMP's */
uchar rgbBmpArray[16][3] = {
0  ,0  , 0  ,
191,0  , 0  ,
0  ,0  , 191,
191,0  , 191,
0  ,191, 0  ,
128,128, 128,
0  ,191, 191,
0  ,0  , 255,
191,191, 0  ,
255,0  , 0  ,
192,192, 192,
255,0  , 255,
0  ,255, 0  ,
255,255, 0  ,
0  ,255, 255,
255,255, 255};

/* this might work with new Win32 16 color BMP's */
uchar rgbXmpArray[16][3] = {
0  , 0  , 0  ,
128, 0  , 0  ,
0  , 0  , 128,
128, 0  , 128,
0  , 128, 0  ,
128, 128, 128,
0  , 128, 128,
0  , 0  , 255,
128, 128, 0  ,
255, 0  , 0  ,
192, 192, 192,
255, 0  , 255,
0  , 255, 0  ,
255, 255, 0  ,
0  , 255, 255,
255, 255, 255};

/* from the bios in some PC I had */
uchar rgbVgaArray[16][3] = {
0  , 0  , 0  ,
255, 0  , 0  ,
0  , 0  , 255,
255, 0  , 255,
0  , 255, 0  ,
85 , 85 , 85 ,
0  , 255, 255,
85 , 85 , 255,
255, 255, 0  ,
255, 85 , 85 ,
192, 192, 192,
255, 85 , 255,
85 , 255, 85 ,
255, 255, 85 ,
85 , 255, 255,
255, 255, 255};

/* some old ZSoft VGA Pcx Colors */
uchar rgbPcxArray[16][3] = {
0  , 0  , 0  ,
170, 0  , 0  ,
0  , 0  , 170,
170, 0  , 170,
0  , 170, 0  ,
85 , 85 , 85 ,
0  , 170, 170,
85 , 85 , 255,
170, 170, 0  ,
255, 85 , 85 ,
170, 170, 170,
255, 85 , 255,
85 , 255, 85 ,
255, 255, 85 ,
85 , 255, 255,
255, 255, 255};

/* ------------------------------------------------------------------- */
/* end of legacy unhacked hack, and back to something that makes sense */
/* ------------------------------------------------------------------- */

/* palette work arrays */
uchar rgbArray[16][3], rgbAppleArray[16][3], rgbPreview[16][3], rgbUser[16][3];
double rgbLuma[16], rgbDouble[16][3];
double rgbLumaBrighten[16], rgbDoubleBrighten[16][3];
double rgbLumaDarken[16], rgbDoubleDarken[16][3];

/* provides base address for page1 hires scanlines  */
unsigned HB[]={
0x2000, 0x2400, 0x2800, 0x2C00, 0x3000, 0x3400, 0x3800, 0x3C00,
0x2080, 0x2480, 0x2880, 0x2C80, 0x3080, 0x3480, 0x3880, 0x3C80,
0x2100, 0x2500, 0x2900, 0x2D00, 0x3100, 0x3500, 0x3900, 0x3D00,
0x2180, 0x2580, 0x2980, 0x2D80, 0x3180, 0x3580, 0x3980, 0x3D80,
0x2200, 0x2600, 0x2A00, 0x2E00, 0x3200, 0x3600, 0x3A00, 0x3E00,
0x2280, 0x2680, 0x2A80, 0x2E80, 0x3280, 0x3680, 0x3A80, 0x3E80,
0x2300, 0x2700, 0x2B00, 0x2F00, 0x3300, 0x3700, 0x3B00, 0x3F00,
0x2380, 0x2780, 0x2B80, 0x2F80, 0x3380, 0x3780, 0x3B80, 0x3F80,
0x2028, 0x2428, 0x2828, 0x2C28, 0x3028, 0x3428, 0x3828, 0x3C28,
0x20A8, 0x24A8, 0x28A8, 0x2CA8, 0x30A8, 0x34A8, 0x38A8, 0x3CA8,
0x2128, 0x2528, 0x2928, 0x2D28, 0x3128, 0x3528, 0x3928, 0x3D28,
0x21A8, 0x25A8, 0x29A8, 0x2DA8, 0x31A8, 0x35A8, 0x39A8, 0x3DA8,
0x2228, 0x2628, 0x2A28, 0x2E28, 0x3228, 0x3628, 0x3A28, 0x3E28,
0x22A8, 0x26A8, 0x2AA8, 0x2EA8, 0x32A8, 0x36A8, 0x3AA8, 0x3EA8,
0x2328, 0x2728, 0x2B28, 0x2F28, 0x3328, 0x3728, 0x3B28, 0x3F28,
0x23A8, 0x27A8, 0x2BA8, 0x2FA8, 0x33A8, 0x37A8, 0x3BA8, 0x3FA8,
0x2050, 0x2450, 0x2850, 0x2C50, 0x3050, 0x3450, 0x3850, 0x3C50,
0x20D0, 0x24D0, 0x28D0, 0x2CD0, 0x30D0, 0x34D0, 0x38D0, 0x3CD0,
0x2150, 0x2550, 0x2950, 0x2D50, 0x3150, 0x3550, 0x3950, 0x3D50,
0x21D0, 0x25D0, 0x29D0, 0x2DD0, 0x31D0, 0x35D0, 0x39D0, 0x3DD0,
0x2250, 0x2650, 0x2A50, 0x2E50, 0x3250, 0x3650, 0x3A50, 0x3E50,
0x22D0, 0x26D0, 0x2AD0, 0x2ED0, 0x32D0, 0x36D0, 0x3AD0, 0x3ED0,
0x2350, 0x2750, 0x2B50, 0x2F50, 0x3350, 0x3750, 0x3B50, 0x3F50,
0x23D0, 0x27D0, 0x2BD0, 0x2FD0, 0x33D0, 0x37D0, 0x3BD0, 0x3FD0};


uchar dhrbuf[16384];

/*

The following is logically reordered to match the lores
color order...

                                                Repeated
                                                Binary
          Color         aux1  main1 aux2  main2 Pattern
          Black          00    00    00    00    0000
          Magenta        08    11    22    44    0001
		  Dark Blue      11    22    44    08    1000
          Violet         19    33    66    4C    1001
          Dark Green     22    44    08    11    0100
          Grey1          2A    55    2A    55    0101
          Medium Blue    33    66    4C    19    1100
          Light Blue     3B    77    6E    5D    1101
          Brown          44    08    11    22    0010
          Orange         4C    19    33    66    0011
          Grey2          55    2A    55    2A    1010
          Pink           5D    3B    77    6E    1011
          Green          66    4C    19    33    0110
          Yellow         6E    5D    3B    77    0111
          Aqua           77    6E    5D    3B    1110
          White          7F    7F    7F    7F    1111

*/

/* the following array is based on the above */
uchar dhrbytes[16][4] = {
	0x00,0x00,0x00,0x00,
	0x08,0x11,0x22,0x44,
	0x11,0x22,0x44,0x08,
	0x19,0x33,0x66,0x4C,
	0x22,0x44,0x08,0x11,
	0x2A,0x55,0x2A,0x55,
	0x33,0x66,0x4C,0x19,
	0x3B,0x77,0x6E,0x5D,
	0x44,0x08,0x11,0x22,
	0x4C,0x19,0x33,0x66,
	0x55,0x2A,0x55,0x2A,
	0x5D,0x3B,0x77,0x6E,
	0x66,0x4C,0x19,0x33,
	0x6E,0x5D,0x3B,0x77,
	0x77,0x6E,0x5D,0x3B,
	0x7F,0x7F,0x7F,0x7F};



/* resizing tables */

/* use as-is to use for 320 x 200 */
/* merge 4 pixels into one to use for 640 x 400 */
uchar mix25to24[24][2] = {
24,1,
23,2,
22,3,
21,4,
20,5,
19,6,
18,7,
17,8,
16,9,
15,10,
14,11,
13,12,
12,13,
11,14,
10,15,
9,16,
8,17,
7,18,
6,19,
5,20,
4,21,
3,22,
2,23,
1,24};

/*

To scale from 320 to 280, a 7 part mix is used from each pixel
in the following pattern with a ratio of 8:7

00000001 11111122 22222333 33334444 44455555 55666666 67777777
00000000 11111111 22222222 33333333 44444444 55555555 66666666

*/

uchar pixel320to280[7][4] = {
7,1,0,1,
6,2,1,2,
5,3,2,3,
4,4,3,4,
3,5,4,5,
2,6,5,6,
1,7,6,7};





/* ***************************************************************** */
/* ========================== code ================================= */
/* ***************************************************************** */


/* intialize the values for the current palette */
void InitDoubleArrays()
{
	int i;
	double dr, dg, db, dthreshold;
	unsigned r, g, b;

    /* array for matching closest color in palette */
	for (i=0;i<16;i++) {
		rgbDouble[i][0] = dr = (double) rgbArray[i][0];
		rgbDouble[i][1] = dg = (double) rgbArray[i][1];
		rgbDouble[i][2] = db = (double) rgbArray[i][2];
		rgbLuma[i] = (dr*299 + dg*587 + db*114) / (255.0*1000);
	}

    /* array for matching closest color in palette
       threshold reduced by 25% */

    if (threshold == 0) {
		dthreshold = 0.75;
	}
	else {
		dthreshold = (double) threshold;
		if (xmatrix != 2) dthreshold *= 0.5;
	    dthreshold = (double) (100.0 - dthreshold) / 100;

	}

	for (i=0;i<16;i++) {
		dr = (double) rgbArray[i][0];
		dg = (double) rgbArray[i][1];
		db = (double) rgbArray[i][2];

		dr *= dthreshold;
		dg *= dthreshold;
		db *= dthreshold;

		rgbDoubleBrighten[i][0] = dr;
		rgbDoubleBrighten[i][1] = dg;
		rgbDoubleBrighten[i][2] = db;
		rgbLumaBrighten[i] = (dr*299 + dg*587 + db*114) / (255.0*1000);
	}

    if (threshold == 0) {
		dthreshold = 1.25;
	}
	else {
		dthreshold = (double) threshold;
		if (xmatrix != 2) dthreshold *= 0.5;
	    dthreshold = (double) (100.0 + dthreshold) / 100;
	}

	for (i=0;i<16;i++) {
		dr = (double) rgbArray[i][0];
		dg = (double) rgbArray[i][1];
		db = (double) rgbArray[i][2];

		dr *= dthreshold;
		if (dr > 255.0) dr = 255.0;
		dg *= dthreshold;
		if (dg > 255.0) dg = 255.0;
		db *= dthreshold;
		if (db > 255.0) db = 255.0;

		rgbDoubleDarken[i][0] = dr;
		rgbDoubleDarken[i][1] = dg;
		rgbDoubleDarken[i][2] = db;
		rgbLumaDarken[i] = (dr*299 + dg*587 + db*114) / (255.0*1000);
	}
}

/* select current palette */
sshort GetBuiltinPalette(sshort palidx, sshort previewidx)
{
	sshort i,j;
	uchar r,g,b;

    printf("Palette %d, Preview Palette %d\n",palidx, previewidx);


	switch(palidx) {
		/* 5 legacy palettes from BMPA2FC */
		/* undocumented, unpromoted, and unfortunate */
		case 11: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbPcxArray[i][0];
					rgbArray[i][1] = rgbPcxArray[i][1];
					rgbArray[i][2] = rgbPcxArray[i][2];
				}
				break;
		case 10: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbVgaArray[i][0];
					rgbArray[i][1] = rgbVgaArray[i][1];
					rgbArray[i][2] = rgbVgaArray[i][2];
				}
				break;
		case 9: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbXmpArray[i][0];
					rgbArray[i][1] = rgbXmpArray[i][1];
					rgbArray[i][2] = rgbXmpArray[i][2];
				}
				break;
		case 8: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbBmpArray[i][0];
					rgbArray[i][1] = rgbBmpArray[i][1];
					rgbArray[i][2] = rgbBmpArray[i][2];
				}
				break;
		case 7: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbCanvasArray[i][0];
					rgbArray[i][1] = rgbCanvasArray[i][1];
					rgbArray[i][2] = rgbCanvasArray[i][2];
				}
				break;

        /* user definable palette file (documented) */
		case 6: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbUser[i][0];
					rgbArray[i][1] = rgbUser[i][1];
					rgbArray[i][2] = rgbUser[i][2];
				}
				break;
        /* documented conversion palettes */
		case 5: for (i=0;i<16;i++) {
					rgbArray[i][0] = grpal[i][0];
					rgbArray[i][1] = grpal[i][1];
					rgbArray[i][2] = grpal[i][2];
				}
				break;

		case 4: for (i=0;i<16;i++) {
					rgbArray[i][0] = wikipedia[i][0];
					rgbArray[i][1] = wikipedia[i][1];
					rgbArray[i][2] = wikipedia[i][2];
				}
				break;

		case 3: for (i=0;i<16;i++) {
					rgbArray[i][0] = awinnewcolors[i][0];
					rgbArray[i][1] = awinnewcolors[i][1];
					rgbArray[i][2] = awinnewcolors[i][2];
				}
				break;
		case 2: for (i=0;i<16;i++) {
					rgbArray[i][0] = awinoldcolors[i][0];
					rgbArray[i][1] = awinoldcolors[i][1];
					rgbArray[i][2] = awinoldcolors[i][2];
				}
				break;
		case 1: for (i=0;i<16;i++) {
					rgbArray[i][0] = ciderpresscolors[i][0];
					rgbArray[i][1] = ciderpresscolors[i][1];
					rgbArray[i][2] = ciderpresscolors[i][2];
				}
				break;
		default: for (i=0;i<16;i++) {
					rgbArray[i][0] = kegs32colors[i][0];
					rgbArray[i][1] = kegs32colors[i][1];
					rgbArray[i][2] = kegs32colors[i][2];
				}
				break;
		}

	switch(previewidx) {
		case 11: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbPcxArray[i][0];
					rgbPreview[i][1] = rgbPcxArray[i][1];
					rgbPreview[i][2] = rgbPcxArray[i][2];
				}
				break;
		case 10: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbVgaArray[i][0];
					rgbPreview[i][1] = rgbVgaArray[i][1];
					rgbPreview[i][2] = rgbVgaArray[i][2];
				}
				break;
		case 9: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbXmpArray[i][0];
					rgbPreview[i][1] = rgbXmpArray[i][1];
					rgbPreview[i][2] = rgbXmpArray[i][2];
				}
				break;
		case 8: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbBmpArray[i][0];
					rgbPreview[i][1] = rgbBmpArray[i][1];
					rgbPreview[i][2] = rgbBmpArray[i][2];
				}
				break;
		case 7: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbCanvasArray[i][0];
					rgbPreview[i][1] = rgbCanvasArray[i][1];
					rgbPreview[i][2] = rgbCanvasArray[i][2];
				}
				break;


		case 6: for (i=0;i<16;i++) {
					rgbPreview[i][0] = rgbUser[i][0];
					rgbPreview[i][1] = rgbUser[i][1];
					rgbPreview[i][2] = rgbUser[i][2];
				}
				break;

		case 5: for (i=0;i<16;i++) {
					rgbPreview[i][0] = grpal[i][0];
					rgbPreview[i][1] = grpal[i][1];
					rgbPreview[i][2] = grpal[i][2];
				}
				break;

		case 4: for (i=0;i<16;i++) {
					rgbPreview[i][0] = wikipedia[i][0];
					rgbPreview[i][1] = wikipedia[i][1];
					rgbPreview[i][2] = wikipedia[i][2];
				}
				break;

		case 3: for (i=0;i<16;i++) {
					rgbPreview[i][0] = awinnewcolors[i][0];
					rgbPreview[i][1] = awinnewcolors[i][1];
					rgbPreview[i][2] = awinnewcolors[i][2];
				}
				break;
		case 2: for (i=0;i<16;i++) {
					rgbPreview[i][0] = awinoldcolors[i][0];
					rgbPreview[i][1] = awinoldcolors[i][1];
					rgbPreview[i][2] = awinoldcolors[i][2];
				}
				break;
		case 1: for (i=0;i<16;i++) {
					rgbPreview[i][0] = ciderpresscolors[i][0];
					rgbPreview[i][1] = ciderpresscolors[i][1];
					rgbPreview[i][2] = ciderpresscolors[i][2];
				}
				break;
		default: for (i=0;i<16;i++) {
					rgbPreview[i][0] = kegs32colors[i][0];
					rgbPreview[i][1] = kegs32colors[i][1];
					rgbPreview[i][2] = kegs32colors[i][2];
				}
				break;
		}



		for (i=0;i<16;i++) {
			rgbAppleArray[i][0] = rgbArray[i][0] >> 4;
			rgbAppleArray[i][1] = rgbArray[i][1] >> 4;
			rgbAppleArray[i][2] = rgbArray[i][2] >> 4;
		}
}

/* use CCIR 601 luminosity to get closest color in current palette */
/* based on palette that has been selected for conversion */
uchar GetMedColor(uchar r, uchar g, uchar b, double *paldistance)
{
	uchar drawcolor;
	double dr, dg, db, diffR, diffG, diffB, luma, lumadiff, distance, prevdistance;
	int i;

    dr = (double)r;
    dg = (double)g;
    db = (double)b;
    luma = (dr*299 + dg*587 + db*114) / (255.0*1000);
    lumadiff = rgbLuma[0]-luma;

	/* Compare the difference of RGB values, weigh by CCIR 601 luminosity */
    /* set palette index to color with shortest distance */

    /* get color distance to first palette color */
	diffR = (rgbDouble[0][0]-dr)/255.0;
	diffG = (rgbDouble[0][1]-dg)/255.0;
	diffB = (rgbDouble[0][2]-db)/255.0;

    prevdistance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         + lumadiff*lumadiff;
    /* set palette index to first color */
    drawcolor = 0;
    paldistance[0] = prevdistance;

    /* get color distance to rest of palette colors */
    for (i=1;i<16;i++) {

        /* get color distance of this index */
		lumadiff = rgbLuma[i]-luma;
		diffR = (rgbDouble[i][0]-dr)/255.0;
		diffG = (rgbDouble[i][1]-dg)/255.0;
		diffB = (rgbDouble[i][2]-db)/255.0;
    	distance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         	+ lumadiff*lumadiff;

        /* if distance is smaller use this index */
	    if (distance < prevdistance) {
		   prevdistance = distance;
		   paldistance[0] = prevdistance;
		   drawcolor = (uchar)i;
		}

	}
	return drawcolor;
}


/* use CCIR 601 luminosity to get closest color in current palette */
/* match values have been decreased by user-defined threshold */
/* brightens darker colors by promoting them */
uchar GetHighColor(uchar r, uchar g, uchar b, double *paldistance)
{
	uchar drawcolor;
	double dr, dg, db, diffR, diffG, diffB, luma, lumadiff, distance, prevdistance;
	int i;

    dr = (double)r;
    dg = (double)g;
    db = (double)b;
    luma = (dr*299 + dg*587 + db*114) / (255.0*1000);
    lumadiff = rgbLumaBrighten[0]-luma;

	/* Compare the difference of RGB values, weigh by CCIR 601 luminosity */
    /* set palette index to color with shortest distance */

    /* get color distance to first palette color */
	diffR = (rgbDoubleBrighten[0][0]-dr)/255.0;
	diffG = (rgbDoubleBrighten[0][1]-dg)/255.0;
	diffB = (rgbDoubleBrighten[0][2]-db)/255.0;

    prevdistance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         + lumadiff*lumadiff;
    /* set palette index to first color */
    drawcolor = 0;
    paldistance[0] = prevdistance;

    /* get color distance to rest of palette colors */
    for (i=1;i<16;i++) {

        /* get color distance of to this index */
		lumadiff = rgbLumaBrighten[i]-luma;
		diffR = (rgbDoubleBrighten[i][0]-dr)/255.0;
		diffG = (rgbDoubleBrighten[i][1]-dg)/255.0;
		diffB = (rgbDoubleBrighten[i][2]-db)/255.0;
    	distance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         	+ lumadiff*lumadiff;

        /* if distance is smaller use this index */
	    if (distance < prevdistance) {
		   prevdistance = distance;
		   paldistance[0] = prevdistance;
		   drawcolor = (uchar)i;
		}

	}
	return drawcolor;
}

/* use CCIR 601 luminosity to get closest color in current palette */
/* match values have been increased by user-defined threshold */
/* darkens lighter colors by demoting them */
uchar GetLowColor(uchar r, uchar g, uchar b, double *paldistance)
{
	uchar drawcolor;
	double dr, dg, db, diffR, diffG, diffB, luma, lumadiff, distance, prevdistance;
	int i;

    dr = (double)r;
    dg = (double)g;
    db = (double)b;
    luma = (dr*299 + dg*587 + db*114) / (255.0*1000);
    lumadiff = rgbLumaDarken[0]-luma;

	/* Compare the difference of RGB values, weigh by CCIR 601 luminosity */
    /* set palette index to color with shortest distance */

    /* get color distance to first palette color */
	diffR = (rgbDoubleDarken[0][0]-dr)/255.0;
	diffG = (rgbDoubleDarken[0][1]-dg)/255.0;
	diffB = (rgbDoubleDarken[0][2]-db)/255.0;

    prevdistance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         + lumadiff*lumadiff;
    /* set palette index to first color */
    drawcolor = 0;
    paldistance[0] = prevdistance;

    /* get color distance to rest of palette colors */
    for (i=1;i<16;i++) {

        /* get color distance of to this index */
		lumadiff = rgbLumaDarken[i]-luma;
		diffR = (rgbDoubleDarken[i][0]-dr)/255.0;
		diffG = (rgbDoubleDarken[i][1]-dg)/255.0;
		diffB = (rgbDoubleDarken[i][2]-db)/255.0;
    	distance = (diffR*diffR*0.299 + diffG*diffG*0.587 + diffB*diffB*0.114)*0.75
         	+ lumadiff*lumadiff;

        /* if distance is smaller use this index */
	    if (distance < prevdistance) {
		   prevdistance = distance;
		   paldistance[0] = prevdistance;
		   drawcolor = (uchar)i;
		}

	}
	return drawcolor;
}

/* switchboard function to handle cross-hatched and non-cross-hatched output */
/* keeps the conditionals out of the main loop */
uchar GetDrawColor(uchar r, uchar g, uchar b, int x, int y)
{

    /* additional vars for future */
    double distance, lowdistance, highdistance;
    uchar drawcolor, lowcolor, highcolor;
    uchar red 	= (uchar)(r >> 4),
		  green = (uchar)(g >> 4),
		  blue 	= (uchar)(b >> 4);
    int i;

	/* quick check for verbatim match */
	for (i = 0; i < 16; i++) {
		if (rgbAppleArray[i][0] == red &&
			rgbAppleArray[i][1] == green &&
			rgbAppleArray[i][2] == blue) return (uchar)i;

	}

    /* non-cross-hatched output */
    if (threshold == 0 && ymatrix == 0) return GetMedColor(r,g,b,&distance);

    if (ymatrix != 0) {
        switch(ymatrix) {
        	case 1: return GetLowColor(r,g,b,&lowdistance);
        	case 3: return GetHighColor(r,g,b,&highdistance);
        	case 2:
        	default:return GetMedColor(r,g,b,&distance);
		}
	}

	/* patterned cross-hatching */
	/* the thresholds are percentage based */
	/* with a user definable threshold */

    switch(xmatrix)
    {
		/* patterns 1, 2, 3 - 2 x 2 patterned cross-hatching */
		case 1:
			/* low, med
			   med, low
			*/
			if (y % 2 == 0) {
				if (x%2 == 1) return GetMedColor(r,g,b,&distance);
				return GetLowColor(r,g,b,&lowdistance);
			}
			if (x%2 == 0) return GetMedColor(r,g,b,&distance);
			return GetLowColor(r,g,b,&lowdistance);

		case 3:
			/* high, med
			   med, high
			*/
			if (y % 2 == 0) {
				if (x%2 == 1) return GetMedColor(r,g,b,&distance);
				return GetHighColor(r,g,b,&highdistance);
			}
			if (x%2 == 0) return GetMedColor(r,g,b,&distance);
			return GetHighColor(r,g,b,&highdistance);

		case 2:
		default:
			/* high, low
			   low, high
			*/
			if (y % 2 == 0) {
				if (x%2 == 1) return GetLowColor(r,g,b,&lowdistance);
				return GetHighColor(r,g,b,&highdistance);
			}
			if (x%2 == 0) return GetLowColor(r,g,b,&lowdistance);
			return GetHighColor(r,g,b,&highdistance);

	}

#ifndef TURBOC
    /* never gets to here */
	return GetMedColor(r,g,b,&distance);
#endif

}

/* routines to save to Apple 2 Double Hires Format */
/* a double hi-res pixel can occur at any one of 7 positions */
/* in a 4 byte block which spans aux and main screen memory */
/* the horizontal resolution is 140 pixels */
void dhrplot(int x,int y,uchar drawcolor)
{
    int xoff, pattern;
    uchar *ptraux, *ptrmain;

    pattern = (x%7);
	xoff = HB[y] + ((x/7) * 2);
    ptraux  = (uchar *) &dhrbuf[xoff-0x2000];
    ptrmain = (uchar *) &dhrbuf[xoff];


	switch(pattern)
	{
		/* left this here for reference

		uchar dhrpattern[7][4] = {
		0,0,0,0,
		0,0,0,1,
		1,1,1,1,
		1,1,2,2,
		2,2,2,2,
		2,3,3,3,
        3,3,3,3};
        */

		case 0: ptraux[0] &= 0x70;
		        ptraux[0] |= dhrbytes[drawcolor][0] &0x0f;
		        break;
		case 1: ptraux[0] &= 0x0f;
		        ptraux[0] |= dhrbytes[drawcolor][0] & 0x70;
		        ptrmain[0] &= 0x7e;
		        ptrmain[0] |= dhrbytes[drawcolor][1] & 0x01;
		        break;
		case 2: ptrmain[0] &= 0x61;
		        ptrmain[0] |= dhrbytes[drawcolor][1] & 0x1e;
		        break;
		case 3: ptrmain[0] &= 0x1f;
		        ptrmain[0] |= dhrbytes[drawcolor][1] & 0x60;
		        ptraux[1] &= 0x7c;
		        ptraux[1] |= dhrbytes[drawcolor][2] & 0x03;
                break;
		case 4: ptraux[1] &= 0x43;
		        ptraux[1] |= dhrbytes[drawcolor][2] & 0x3c;
		        break;
		case 5: ptraux[1] &= 0x3f;
		        ptraux[1] |= dhrbytes[drawcolor][2] & 0x40;
 		        ptrmain[1] &= 0x78;
 		        ptrmain[1] |= dhrbytes[drawcolor][3] & 0x07;
 		        break;
		case 6: ptrmain[1] &= 0x07;
		        ptrmain[1] |= dhrbytes[drawcolor][3] & 0x78;
 		        break;
	}

}


/* monochrome DHGR - 560 x 192 */
unsigned char dhbmono[] = {0x7e,0x7d,0x7b,0x77,0x6f,0x5f,0x3f};
unsigned char dhwmono[] = {0x1,0x2,0x4,0x8,0x10,0x20,0x40};

int dhrmonoplot(int x, int y, uchar drawcolor)
{

    int xoff, pixel;
    uchar *ptr;

    xoff = HB[y] + (x/14);
    pixel = (x%14);
    if (pixel > 6) {
		/* main memory */
		pixel -= 7;
		ptr = (uchar *) &dhrbuf[xoff];
	}
	else {
		/* auxiliary memory */
		ptr  = (uchar *) &dhrbuf[xoff-0x2000];
	}

	if (drawcolor != 0) {
		/* white */
		ptr[0] |= dhwmono[pixel]; /* inclusive OR */
	}
	else {
		/* black */
		ptr[0] &= dhbmono[pixel]; /* bitwise AND */
	}

}


void dhrfill(int y,uchar drawcolor)
{
    int xoff, x;
    uchar *ptraux, *ptrmain;

	xoff = HB[y];

    ptraux  = (uchar *) &dhrbuf[xoff-0x2000];
    ptrmain = (uchar *) &dhrbuf[xoff];

    for (x = 0,xoff=0; x < 20; x++) {
		ptraux[xoff]  = dhrbytes[drawcolor][0];
		ptrmain[xoff] = dhrbytes[drawcolor][1]; xoff++;
		ptraux[xoff]  = dhrbytes[drawcolor][2];
		ptrmain[xoff] = dhrbytes[drawcolor][3]; xoff++;
	}
}


/* initialize the scanlines in the write buffer
   to the background color

   this doesn't matter for a full-screen image

*/
void dhrclear()
{
	int y;
	uchar drawcolor;
	memset(dhrbuf,0,16384);
	if (backgroundcolor == LOBLACK) return;
	drawcolor = (uchar)backgroundcolor;
	for (y=0;y<192;y++) dhrfill(y,drawcolor);
}

/* mono-spaced "tom thumb" 4 x 6 font */
/* using a byte map to gain a little speed at the expense of memory */
/* a bitmap could have been encoded into nibbles of 3 bytes per character
   rather than the 18 bytes per character that I am using
   but the trade-off in the speed in unmasking would have slowed this down */
void plotthumbDHGR(unsigned char ch, unsigned x, unsigned y,
               unsigned char fg, unsigned char bg)
{
	unsigned offset, x1, x2=x+3, y2=y+6;
	unsigned char byte;

	if (ch < 33 || ch > 127) ch = 0;
	else ch -=32;

	if (ch == 0 && bg > 15) return;

    /* each of the 96 characters is encoded into 18 bytes */
	offset = (18 * ch);

    while (y < y2) {
		for (x1 = x; x1 < x2; x1++) {
		   if (x1 > 139) {
			   offset++;
			   continue;
		   }

		   byte = tomthumb[offset++];

		   if (byte == 0) {
			   if (bg > 15) continue;
			   dhrplot(x1,y,bg);
		   }
		   else {
			   if (fg > 15) continue;
			   dhrplot(x1,y,fg);
	   	   }
		}
		/* if background color is being used then a trailing pixel is required
		   between characters */
		if (bg < 16 && x2 < 140)dhrplot(x2,y,bg);

		if (y++ > 191) break;
	}

}

/* normally spaced 4 x 6 font */
/* using character plotting function plotthumb() (above) */
void thumbDHGR(char *str,unsigned x, unsigned y,
              unsigned char fg,unsigned char bg, unsigned char justify)
{
  int target;
  unsigned char ch;

  if (justify == 'M' || justify == 'm') {
	 target = strlen(str);
	 x-= ((target * 4) /2);
  }

  while ((ch = *str++) != 0) {
	 plotthumbDHGR(ch,x,y,fg,bg);
	 x+=4;
  }
}


/* VBMP output routines */
/* VBMP requires a palette in the DHGR color order

   My Apple II routines whether in cc65, Aztec C65, or in my converters
   always use the LORES color order. LORES, Double LORES, and DHGR all use
   the same colors so it seems rather silly to use a different index value
   for DHGR in re-usable program code.

   However, since VBMP doesn't really care about the color palette in a BMP
   and just the order of the palette, we need to remap the palette and the
   scaline palette indices to DHGR color order.


*/

/* http://en.wikipedia.org/wiki/List_of_8-bit_computer_hardware_palettes */
uchar rgbVBMP[16][3] = {
	0,0,0,		 /* black */
	114,38,64,   /* red */
	64,51,127,   /* dk blue */
	228,52,254,  /* purple */
	14,89,64,    /* dk green */
	128,128,128, /* gray */
	27,154,254,  /* med blue */
	191,179,255, /* lt blue */
	64,76,0,     /* brown */
	228,101,1,   /* orange */
	128,128,128, /* grey */
	241,166,191, /* pink */
	27,203,1,    /* lt green */
	191,204,128, /* yellow */
	141,217,191, /* aqua */
	255,255,255};/* white */




unsigned char RemapLoToHi[16] = {
	LOBLACK,
	LORED,
	LOBROWN,
	LOORANGE,
	LODKGREEN,
	LOGRAY,
	LOLTGREEN,
	LOYELLOW,
	LODKBLUE,
	LOPURPLE,
	LOGREY,
	LOPINK,
	LOMEDBLUE,
	LOLTBLUE,
	LOAQUA,
	LOWHITE};

/* 560 x 192 - verbatim mono output format */
unsigned char mono192[62] ={
0x42, 0x4D, 0x3E, 0x36, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x30, 0x02, 0x00, 0x00, 0xC0, 0x00,
0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00};

ushort WriteVbmpHeader(FILE *fp)
{
    ushort outpacket;
    int c, i, j;

    /* BMP scanlines are padded to a multiple of 4 bytes (DWORD) */
    outpacket = (ushort)72;

    if (mono != 0) {
		c = fwrite(mono192,1,sizeof(mono192),fp);
		if (c!= sizeof(mono192))return 0;
        return outpacket;
	}

    memset((char *)&mybmp.bfi.bfType[0],0,sizeof(BMPHEADER));

    /* create the info header */
    mybmp.bmi.biSize = (ulong)40;
    mybmp.bmi.biWidth  = (ulong)140;
    mybmp.bmi.biHeight = (ulong)192;
    mybmp.bmi.biPlanes = 1;
    mybmp.bmi.biBitCount = 4;
    mybmp.bmi.biCompression = (ulong) BI_RGB;

    mybmp.bmi.biSizeImage = (ulong)outpacket;
	mybmp.bmi.biSizeImage *= mybmp.bmi.biHeight;

    /* create the file header */
    mybmp.bfi.bfType[0] = 'B';
    mybmp.bfi.bfType[1] = 'M';
    mybmp.bfi.bfOffBits = (ulong) sizeof(BMPHEADER) + sizeof(RGBQUAD) * 16;
    mybmp.bfi.bfSize = mybmp.bmi.biSizeImage + mybmp.bfi.bfOffBits;

 	/* write the header for the output BMP */
    c = fwrite((char *)&mybmp.bfi.bfType[0],sizeof(BMPHEADER),1,fp);

    if (c!= 1)return 0;

    /* use the preview palette for the VBMP palette for now */
    /* not sure if this will work */
    for (i=0;i<16;i++) {
		j = RemapLoToHi[i];
		sbmp[i].rgbRed   = rgbVBMP[j][RED];
		sbmp[i].rgbGreen = rgbVBMP[j][GREEN];
    	sbmp[i].rgbBlue  = rgbVBMP[j][BLUE];

	}

	/* write the palette for the output bmp */
	c = fwrite((char *)&sbmp[0].rgbBlue, sizeof(RGBQUAD)*16,1,fp);
	if (c!= 1)return 0;

return outpacket;
}

/* write 140 x 192 x 16 color bmp */
int WriteVBMPFile()
{

    FILE *fp;
    uchar ch;
	int x,x1,y,y2,idx,j;

	fp = fopen(vbmpfile,"wb");

	if (fp == NULL) {
		printf("Error opening %s for writing!\n",vbmpfile);
		return INVALID;
	}

	if (WriteVbmpHeader(fp) == 0) {
		fclose(fp);
		remove(vbmpfile);
		printf("Error writing header to %s!\n",vbmpfile);
		return INVALID;
	}
	memset(&bmpscanline[0],0,72);

    /* write 4 bit packed scanlines */
    /* remap from LORES color order to DHGR color order */

    y2 = 191;
   	for (y = 0; y< 192; y++) {

	   for (x = 0, x1=0; x < 140; x++) {
		  if (x%2 == 0) {
		  	idx = dhrgetpixel(x,y2);
          	/* range check */
          	if (idx < 0 || idx > 15)idx = 0; /* default black */
          	j = RemapLoToHi[idx];
          	ch = (uchar)j << 4;
	      }
	      else {
		  	idx = dhrgetpixel(x,y2);
          	/* range check */
          	if (idx < 0 || idx > 15)idx = 0; /* default black */
          	j = RemapLoToHi[idx];
          	bmpscanline[x1] = ch | (uchar)j; x1++;
		  }

	   }
	   fwrite((char *)&bmpscanline[0],1,72,fp);
	   y2 -= 1;
    }

    fclose(fp);
    if (quietmode == 1)printf("%s created!\n",vbmpfile);
    return SUCCESS;

}

/* save both raw output file formats */
int savedhr()
{

	FILE *fp;
	int c;

    if (outputtype != BIN_OUTPUT) return SUCCESS;

    GetUserTextFile();

    if (applesoft == 0) {

		fp = fopen(a2fcfile,"wb");
		if (NULL == fp) {
	    	if (quietmode == 1)printf("Error Opening %s for writing!\n",a2fcfile);
			return INVALID;
		}
		c = fwrite(dhrbuf,1,16384,fp);
		fclose(fp);

		if (c != 16384) {
			remove(a2fcfile);
			if (quietmode == 1)printf("Error Writing %s!\n",a2fcfile);
			return INVALID;
		}
		if (quietmode == 1)printf("%s created!\n",a2fcfile);
		if (vbmp != 0) {
			WriteVBMPFile();
		}
		return SUCCESS;
	}


    /* the bsaved images are split into two files
       the first file is loaded into aux mem */
   	fp = fopen(auxfile,"wb");
	if (NULL == fp) {
	    if (quietmode == 1)printf("Error Opening %s for writing!\n",auxfile);
		return INVALID;
	}
	c = fwrite(dhrbuf,1,8192,fp);
	fclose(fp);
	if (c != 8192) {
		remove(auxfile);
		if (quietmode == 1)printf("Error Writing %s!\n",auxfile);
		return INVALID;
	}

    /* the second file is loaded into main mem */
	fp = fopen(mainfile,"wb");
	if (NULL == fp) {
		remove(auxfile);
		if (quietmode == 1)printf("Error Opening %s for writing!\n",mainfile);
		return INVALID;
	}
	c = fwrite(&dhrbuf[8192],1,8192,fp);
	fclose(fp);
	if (c != 8192) {
		/* remove both files */
		remove(auxfile);
		remove(mainfile);
		if (quietmode == 1)printf("Error Writing %s!\n",mainfile);
		return INVALID;
	}

	if (quietmode == 1) {
		printf("%s created!\n",auxfile);
		printf("%s created!\n",mainfile);
	}

	if (vbmp != 0) {
		WriteVBMPFile();
	}

	return SUCCESS;
}

/* save raster oriented DHGR image fragment

   file format is 2 byte header

   1 byte - width in bytes (multiples of 4 bytes - 7 pixels)
   1 byte - height in rasters

   followed by interleaved raster data

   aux raster, main raster = (width in bytes)
   aux raster, main raster = (width in bytes)
   aux raster, main raster = (width in bytes)
   etc...

*/
int savesprite()
{

	FILE *fp;
	int i, c, width, packet, y, xoff, cnt;
	uchar *ptraux, *ptrmain, ch;

    if (outputtype != SPRITE_OUTPUT) return SUCCESS;

    if (scale == 1) spritewidth = bmpwidth / 2;
    else spritewidth = bmpwidth;

    if (spritewidth < 1) {
	   if (quietmode == 1)printf("Width is too small for %s!\n",spritefile);
	   return INVALID;
    }
    while (spritewidth%7 != 0) spritewidth++;

    width = (int)((spritewidth / 7) * 4); /* 4 bytes = 7 pixels */
    packet = (int)width / 2;

	fp = fopen(spritefile,"wb");
	if (NULL == fp) {
	    if (quietmode == 1)printf("Error Opening %s for writing!\n",spritefile);
		return INVALID;
	}

	/* 2 byte header */
	fputc((uchar)width,fp);          /* width in bytes */
	fputc((uchar)bmpheight,fp);      /* height in scanlines */

     /* write header values to stdout */
	if (quietmode == 0) {
		printf("#define %sWIDTH  %d\n",fname,width);
		printf("#define %sHEIGHT %d\n",fname,bmpheight);
		printf("#define %sSIZE   %d\n\n",fname,width * bmpheight);
		printf("uchar %sBackgroundColor = %d;\n\n",fname,backgroundcolor);
		printf("/* Embedded DHGR Image Fragment created from %s */\n\n",bmpfile);
        printf("uchar %sPixelData[] = {\n",fname);
	}


	for (y = 0, cnt = 0; y < bmpheight; y++) {
		xoff = HB[y];
    	ptraux  = (uchar *) &dhrbuf[xoff-0x2000];
    	ptrmain = (uchar *) &dhrbuf[xoff];
    	/* aux raster */
        c = fwrite((char *)&ptraux[0],1,packet,fp);
        if (c!= packet) break;
        /* main raster */
        c = fwrite((char *)&ptrmain[0],1,packet,fp);
        if (c!= packet) break;

		if (quietmode == 0) {
		    for (i=0;i<width;i++) {
				if (i <packet)ch = ptraux[i];
				else ch = ptrmain[i-packet];
				if (cnt == 0) {
					printf("%3d",ch);
				}
				else {
					printf(",");
					if (cnt%16 == 0) printf("\n");
					printf("%3d",ch);
				}
				cnt++;
			}
		}
	}
	fclose(fp);

	if (c!=packet) {
		remove(spritefile);
	    if (quietmode == 1)printf("Error Writing %s!\n",spritefile);
	    return INVALID;
	}

	if (quietmode == 1)printf("%s created!\n",a2fcfile);
	else printf("};\n\n");

	return SUCCESS;
}


/* read and remap a mask line from an open mask file */
/* required by dithered and non-dithered routines when in use */
sshort ReadMaskLine(ushort y)
{
	ulong pos;
	ushort x;
	uchar ch;

	if (overlay == 0) return INVALID;

	pos = (ulong) (191 - y);
    pos *= 140;
    pos += maskbmp.bfi.bfOffBits;

    fseek(fpmask,pos,SEEK_SET);
	fread((char *)&maskline[0],1,140,fpmask);
	for (x = 0; x < 140; x++) {
		ch = maskline[x];
		maskline[x] = remap[ch];
	}
	return SUCCESS;
}


/*

http://www.efg2.com/Lab/Library/ImageProcessing/DHALF.TXT


The Floyd-Steinberg filter

This is where it all began, with Floyd and Steinberg's pioneering
research in 1975.  The filter can be diagrammed thus:


          *   7
      3   5   1     (1/16)


In this (and all subsequent) filter diagrams, the "*" represents the pixel
currently being scanning, and the neighboring numbers (called weights)
represent the portion of the error distributed to the pixel in that
position.  The expression in parentheses is the divisor used to break up the
error weights.  In the Floyd-Steinberg filter, each pixel "communicates"
with 4 "neighbors."  The pixel immediately to the right gets 7/16 of the
error value, the pixel directly below gets 5/16 of the error, and the
diagonally adjacent pixels get 3/16 and 1/16.

The weighting shown is for the traditional left-to-right scanning of the
image.  If the line were scanned right-to-left (more about this later), this
pattern would be reversed.  In either case, the weights calculated for the
subsequent line must be held by the program, usually in an array of some
sort, until that line is visited later.

Floyd and Steinberg carefully chose this filter so that it would produce a
checkerboard pattern in areas with intensity of 1/2 (or 128, in our sample
image).  It is also fairly easy to execute in programming code, since the
division by 16 is accomplished by simple, fast bit-shifting instructions
(this is the case whenever the divisor is a power of 2).

*/

/*

Floyd-Steinberg dithering published by Robert Floyd and Louis Steinberg in
1976 was the first 2D error diffusion dithering formula.(Filter Lite is an
algorithm by Sierra that produces similar results.) Floyd-Steinberg dithering
only diffuses the error to neighbouring pixels. This results in very
fine-grained dithering.

In the same year a much more powerful algorithm was also published: Jarvis,
Judice, and Ninke. (Sierra dithering and Sierra 2 are based on Jarvis
dithering, and produce similar results. Atkinson dithering resembles Jarvis
dithering and Sierra dithering; speckling is improved but very light and dark
areas may appear blown out.) Jarvis dithering is coarser than Floyd-Steinberg,
but has fewer visual artifacts.

Five years after Jarvis dithering, Peter Stucki published an adjusted version,
to improve processing time. Its output tends to be clean and sharp.

Seven years after Stucki published his improvement to Jarvis, Judice, Ninke
dithering, Daniel Burkes developed a simplified form of Stucki dithering that
is somewhat less clean and sharp.

*/

int globalclip = 0;

/* setting clip to 0 increases the potential amount of retained error */
/* error is accumulated in a short integer and may be negative or positive */
uchar AdjustShortPixel(int clip,sshort *buf,sshort value)
{

    if (globalclip == 1) clip = 1;

    value = (sshort)(buf[0] + value);
    if (clip != 0) {
    	if (value < 0) value = 0;
    	else if (value > 255) value = 255;
	}
    buf[0] = value;
   	if (clip == 0) {
    	if (value < 0) value = 0;
    	else if (value > 255) value = 255;
	}
    return (uchar) value;
}

int ditherstart = 0;
int bleed = 16;


/* custom dither file support */
sshort customdivisor;
sshort customdither[3][11];

/* helper function for ReadCustomDither */
int InitCustomLine(char *ptr, int lidx)
{
	int cnt=0, i;

    customdither[lidx][cnt] = (sshort) atoi(ptr);

    /* enforce 11 fields */
	for (i=0;ptr[i]!=0;i++) {
		if (ptr[i]== ',') {
			cnt++;
			if (cnt < 11) customdither[lidx][cnt] = (sshort) atoi((char*)&ptr[i+1]);
		}
	}
	if (cnt != 10) return -1;
}


/* read a custom dither pattern from a comma delimited text file

   line 1 is the custom divisor
   the next 3 lines are 11 fields in the following format:

	0,0,0,0,0,*,0,0,0,0,0
	0,0,0,0,0,0,0,0,0,0,0
	0,0,0,0,0,0,0,0,0,0,0

	unused fields must be padded with zeros

	errata:

	- no range checking
    - current pixel (asterisk) at subscript 5 is not "protected"

*/
int ReadCustomDither(char *name)
{
	FILE *fp;
	char ch, buf[128];
	int i,j;

    /* clear 3-dimensional custom dither array */
    memset(&customdither[0][0],0,sizeof(sshort)*33);

	fp = fopen(name,"r");
	if (NULL == fp) return -1;

	/* read divisor */
	for (;;) {
		if (NULL == fgets(buf, 128, fp)) {
	    	fclose(fp);
	    	return -1;
		}
		/* ignore comment lines and blank lines */
		ch = buf[0];
		/* leading numeric characters only */
		if (ch < 48 || ch > 57) continue;
		break;
	}
	customdivisor = (sshort) atoi(buf);
	if (customdivisor < 1) {
		fclose(fp);
		return -1;
	}

    /* read up to 3 lines of dither pattern */
	for (i=0;;) {
		if (NULL == fgets(buf, 128, fp)) {
	    	fclose(fp);
	    	return -1;
		}
		/* ignore comment lines and blank lines */
		ch = buf[0];
		/* leading numeric characters only */
		if (ch < 48 || ch > 57) continue;
		/* condition line - remove trailing comments */
		for (j=0;buf[j]!=0;j++) {
			ch = buf[j];
			/* numeric characters are ok */
			if (ch > 47 && ch < 58) continue;
			/* commas and asterisks are ok */
			if (ch == ',' || ch == '*') continue;
			buf[j] = 0;
			break;
		}

        /* parse fields - there must be 11 fields */
		if(InitCustomLine((char *)&buf[0],i)==-1) {
			fclose(fp);
			return -1;
		}
		i++;
		if (i == 3) break;
	}
	fclose(fp);
	if (i == 0) return -1;

	if (quietmode == 1) {
		printf("Imported Dither from %s\n",name);
	}

    dither = CUSTOM;
    return SUCCESS;

}


/* http://en.wikipedia.org/wiki/Floyd%E2%80%93Steinberg_dithering */
/* http://www.tannerhelland.com/4660/dithering-eleven-algorithms-source-code/ */
/* http://www.efg2.com/Lab/Library/ImageProcessing/DHALF.TXT */
void FloydSteinberg(int y, int width)
{

	double paldistance; /* not used in this function */
	sshort red, green, blue, red_error, green_error, blue_error;
	sshort pos, mult;
    int dx,i,x,x1, total_difference, total_error, total_used;
    uchar drawcolor, r,g,b;

   if (ditherstart == 0) {
	   if (quietmode == 1) {
		  if (mono == 1) puts("Monochrome Dithered Output:");
		  else puts("Color Dithered Output:");

		  if (colorbleed < 100)
		   	printf("Dither = %d - %s, Color Bleed Increase: %d%%\n",dither,dithertext[dither-1],(colorbleed-100)*-1);
		  else if (colorbleed > 100)
		  	printf("Dither = %d - %s, Color Bleed Reduction: %d%%\n",dither,dithertext[dither-1],(colorbleed-100));
		  else
		    printf("Dither = %d - %s\n",dither,dithertext[dither-1]);

		  if (serpentine == 1) puts("Serpentine effect is on!");

	   }
	   ditherstart = 1;
	   /* reduce or increase color bleed */
	   switch(dither) {
			case  FLOYDSTEINBERG: 	bleed = (16 * colorbleed)/100; break;
			case  JARVIS:			bleed = (48 * colorbleed)/100; break;
			case  STUCKI:			bleed = (42 * colorbleed)/100; break;
			case  ATKINSON:         bleed = (8  * colorbleed)/100; break;
			case  BURKES:
			case  SIERRA:           bleed = (32 * colorbleed)/100; break;
			case  SIERRATWO:        bleed = (16 * colorbleed)/100; break;
			case  SIERRALITE:       bleed = (4  * colorbleed)/100; break;
			case  CUSTOM:           bleed = (customdivisor * colorbleed)/100; break;
			default:				bleed = (8  * colorbleed)/100; break; /* same as atkinson */
		}
		if (bleed < 1) bleed = 1;
   }

   /* from left to right */
   for (x=0;x<width;x++) {

      red   = redDither[x];
      green = greenDither[x];
      blue  = blueDither[x];

      r = (uchar)red;
      g = (uchar)green;
      b = (uchar)blue;

      drawcolor = GetDrawColor(r,g,b,x,y);

	  r = rgbArray[drawcolor][RED];
	  g = rgbArray[drawcolor][GREEN];
	  b = rgbArray[drawcolor][BLUE];

      redDither[x]   = (int)r;
      greenDither[x] = (int)g;
      blueDither[x]  = (int)b;

      /* the error is linear in this implementation */
      /* - an integer is used so round-off of errors occurs
      	 - also clipping of the error occurs under some circumstances
      	 - no luminance consideration
      	 - no gamma correction
      */

      red_error   = red - r;
      green_error = green - g;
      blue_error  = blue - b;


	for (i=0;i<3;i++) {

        /* loop through all 3 RGB channels */
		switch(i) {
			case RED:   colorptr = (sshort *)&redDither[0];
						seedptr   = (sshort *)&redSeed[0];
						seed2ptr  = (sshort *)&redSeed2[0];
						color_error = red_error;
						break;
			case GREEN: colorptr = (sshort *)&greenDither[0];
						seedptr   = (sshort *)&greenSeed[0];
						seed2ptr  = (sshort *)&greenSeed2[0];
						color_error = green_error;
						break;
			case BLUE:  colorptr = (sshort *)&blueDither[0];
						seedptr   = (sshort *)&blueSeed[0];
						seed2ptr  = (sshort *)&blueSeed2[0];
						color_error = blue_error;
						break;
		}

        /* diffuse the error based on the dither */
		switch(dither) {
			/* F 1*/
			case FLOYDSTEINBERG:
				/*
					*   7
				3   5   1 	(1/16)

                Serpentine

                7   *
                1   5   3

				*/

                /* if error summing is turned-on add the accumulated rounding error
                   to the next pixel */
                if (errorsum == 0) {
					total_difference = 0;
				}
				else {
					total_error = (color_error * 16) / bleed;
					total_used =  (color_error * 3)/bleed;
					total_used += (color_error * 5)/bleed;
					total_used += (color_error * 1)/bleed;
					total_used += (color_error * 7)/bleed;
					total_difference = total_error - total_used;
				}

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
                	/* finish this line */
                	/* for serpentine effect line 1 error is added behind */
					if (x > 0) AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error * 7)/bleed)+total_difference);
					/* seed next line forward */
					/* for serpentine effect line 2 error is reversed */
					if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 1)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 3)/bleed));

				}
				else {
                	/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 7)/bleed)+total_difference);
					/* seed next line forward */
					if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 3)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 1)/bleed));
				}

				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error * 5)/bleed));
				break;

			/* J 2 */
			case JARVIS:
				/*
					*   7   5
				3   5   7   5   3
				1   3   5   3   1	(1/48)
				*/


				/* finish this line */
				AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 7)/bleed));
				AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)((color_error * 5)/bleed));

				/* seed next lines forward */
				if (x>0){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 5)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seed2ptr[x-1],(sshort)((color_error * 3)/bleed));
				}
				if (x>1){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-2],(sshort)((color_error * 3)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seed2ptr[x-2],(sshort)(color_error/bleed));

				}

				/* seed next line forward */
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error * 7)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 5)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+2],(sshort)((color_error * 3)/bleed));

				/* seed furthest line forward */
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x],(sshort)((color_error * 5)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x+1],(sshort)((color_error * 3)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x+2],(sshort)(color_error/bleed));
				break;

			/* S 3 */
			case STUCKI:
				/*
						*   8   4
				2   4   8   4   2
				1   2   4   2   1	(1/42)
          		*/

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if(x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error * 8)/bleed));
					if(x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)((color_error * 4)/bleed));

				}
                else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 8)/bleed));
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)((color_error * 4)/bleed));
				}

				/* seed next lines forward */
				if (x>0){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 4)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seed2ptr[x-1],(sshort)((color_error * 2)/bleed));
				}
				if (x>1){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-2],(sshort)((color_error * 2)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seed2ptr[x-2],(sshort)(color_error/bleed));

				}

				/* seed next line forward */
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error * 8)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 4)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+2],(sshort)((color_error * 2)/bleed));

				/* seed furthest line forward */
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x],(sshort)((color_error * 4)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x+1],(sshort)((color_error * 2)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x+2],(sshort)(color_error/bleed));
				break;

            /* A 4 */
            case ATKINSON:
            	/*
					*   1   1
				1   1   1
					1			(1/8)

				*/

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if (x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)(color_error/bleed));
					if (x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)(color_error/bleed));
				}
				else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)(color_error/bleed));
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)(color_error/bleed));
				}


				/* seed next line forward */
				if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)(color_error/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)(color_error/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)(color_error/bleed));

				/* seed furthest line forward */
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x],(sshort)(color_error/bleed));
				break;

			/* B 5 */
			case BURKES:
				/*
						*   8   4
				2   4   8   4   2	(1/32)
				*/

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if(x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error * 8) /bleed));
					if(x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)((color_error * 4) /bleed));

				}
                else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 8) /bleed));
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)((color_error * 4) /bleed));

				}

				/* seed next line forward */
				if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 4) / bleed));
				if (x>1)AdjustShortPixel(threshold,(sshort *)&seedptr[x-2],(sshort)((color_error * 2) / bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error * 8) /bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 4) /bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+2],(sshort)((color_error * 2) /bleed));
				break;

            /* SI 6 */
			case SIERRA:
				/*
        	    		*   5   3
     			2   4   5   4   2
         			2   3   2		(1/32)
         		*/
                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if(x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error * 5)/bleed));
					if(x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)((color_error * 3)/bleed));
				}
                else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 5)/bleed));
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)((color_error * 3)/bleed));
				}

				/* seed next lines forward */
				if (x>0){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error * 4)/bleed));
					AdjustShortPixel(threshold,(sshort *)&seed2ptr[x-1],(sshort)((color_error * 2)/bleed));
				}
				if (x>1){
					AdjustShortPixel(threshold,(sshort *)&seedptr[x-2],(sshort)((color_error * 2)/bleed));
				}

				/* seed next line forward */
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error * 5)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error * 4)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+2],(sshort)((color_error * 2)/bleed));

				/* seed furthest line forward */
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x],(sshort)((color_error * 3)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x+1],(sshort)((color_error * 2)/bleed));
				break;

            /* S2 7 */
			case SIERRATWO:
				/*
						*   4   3
			    1   2   3   2   1	(1/16)
			    */

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if(x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error*4)/bleed));
					if(x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)((color_error*3)/bleed));
				}
                else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error*4)/bleed));
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)((color_error*3)/bleed));
				}

				/* seed next line forward */
				if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)((color_error*2)/bleed));
				if (x>1)AdjustShortPixel(threshold,(sshort *)&seedptr[x-2],(sshort)(color_error/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error*3)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)((color_error*2)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+2],(sshort)(color_error/bleed));
				break;

			/* SL 8 */
			case SIERRALITE:
                /*
        	 		*   2
     			1   1		(1/4)
     			*/

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
  					/* finish this line */
					if (x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error * 2) /bleed));

					/* seed next line forward */
					AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)(color_error/bleed));
				}
                else {
					/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error * 2) /bleed));

					/* seed next line forward */
					if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)(color_error/bleed));
				}
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)(color_error/bleed));

				break;


           case CUSTOM:

                /* 0,0,0,0,0,*,0,0,0,0,0
                   0,0,0,0,0,0,0,0,0,0,0
                   0,0,0,0,0,0,0,0,0,0,0 */

                for (dx = 0,pos=x-5;dx < 11; dx++,pos++) {
				   /* finish this line */
				   if (pos < 0) continue;

				   mult = customdither[0][dx];
				   if (mult > 0) {
					   AdjustShortPixel(1,(sshort *)&colorptr[pos],(sshort)((color_error * mult) /bleed));
				   }
				   /* seed next line forward */
				   mult = customdither[1][dx];
				   if (mult > 0) {
					   AdjustShortPixel(threshold,(sshort *)&seedptr[pos],(sshort)((color_error * mult) /bleed));
				   }
				   /* seed furthest line forward */
				   mult = customdither[2][dx];
				   if (mult > 0) {
					   AdjustShortPixel(threshold,(sshort *)&seed2ptr[pos],(sshort)((color_error * mult) /bleed));
				   }

			    }
			    break;

			default:
			   /*
				  * 2 1
				1 2 1
				  1          (1/8)

				Serpentine

			  1 2 *
				1 2 1
		          1



				*/

                /* if error summing is turned-on add the accumulated rounding error
                   to the next pixel */
                if (errorsum == 0) {
					total_difference = 0;
				}
				else {
					total_error = (color_error * 8) / bleed;
					total_used =  (color_error * 2)/bleed;
					total_used += (color_error * 2)/bleed;
					total_used += (color_error /bleed);
					total_used += (color_error /bleed);
					total_used += (color_error /bleed);
					total_used += (color_error /bleed);
					total_difference = total_error - total_used;
				}

                /* for serpentine effect alternating scanlines run the error in reverse */
				if (serpentine == 1 && y%2 == 1) {
					/* finish this line */
					if (x>0)AdjustShortPixel(1,(sshort *)&colorptr[x-1],(sshort)((color_error*2)/bleed)+total_difference);
					if (x>1)AdjustShortPixel(1,(sshort *)&colorptr[x-2],(sshort)(color_error/bleed));
				}
				else {
			    	/* finish this line */
					AdjustShortPixel(1,(sshort *)&colorptr[x+1],(sshort)((color_error*2)/bleed)+total_difference);
					AdjustShortPixel(1,(sshort *)&colorptr[x+2],(sshort)(color_error/bleed));
				}

				/* seed next line forward */
				if (x>0)AdjustShortPixel(threshold,(sshort *)&seedptr[x-1],(sshort)(color_error/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x],(sshort)((color_error*2)/bleed));
				AdjustShortPixel(threshold,(sshort *)&seedptr[x+1],(sshort)(color_error/bleed));

				/* seed furthest line forward */
				AdjustShortPixel(threshold,(sshort *)&seed2ptr[x],(sshort)(color_error/bleed));



			}
		}
	}

   /* get the mask line from the mask file if we are overlaying this image */
   /* the mask file is always a 140 x 192 x 256 color BMP and is applied after rendering is complete */
   /* immediately before Preview files are written to disk and the DHGR buffer is plotted */
   if (overlay == 1) {
   		ReadMaskLine(y);
   }

   /* plot dithered scanline in DHGR buffer using selected conversion palette */
   /* plot dithered scanline in Preview buffer using selected preview palette */
   for (x=0,x1=0;x<width;x++) {


        maskpixel = 0;
        if (overlay == 1) {
			overcolor = maskline[x];
			/* clearcolor is the transparent color for the mask */
			/* if the overlay color is some other color then the pixel is overlaid
			   with the mask color */
			if (overcolor != clearcolor) maskpixel = 1;
		}

        if (maskpixel == 1) {
			drawcolor = (uchar)overcolor;
		}
		else {
			r = (uchar)redDither[x];
			g = (uchar)greenDither[x];
			b = (uchar)blueDither[x];
			drawcolor = GetMedColor(r,g,b,&paldistance);
		}
		if (mono == 1) dhrmonoplot(x,y,drawcolor);
		else dhrplot(x,y,drawcolor);

		/* if color preview option, plot double-wide pixels in pairs of 24-bit RGB triples */
		if (preview == 1) {
			if (mono == 1) {
				previewline[x1] = rgbPreview[drawcolor][BLUE]; x1++;
				previewline[x1] = rgbPreview[drawcolor][GREEN];x1++;
				previewline[x1] = rgbPreview[drawcolor][RED];  x1++;

			}
			else {
				/* we are plotting a double pixel in a 6 byte chunk - b,g,r,b,g,r */
				previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][BLUE]; x1++;
				previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][GREEN];x1++;
				previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][RED];  x1+=4;
			}
		}
   }
}

ushort WriteDIBHeader(FILE *fp, ushort pixels, ushort rasters)
{
    ushort outpacket;
    int c;

    memset((char *)&mybmp.bfi.bfType[0],0,sizeof(BMPHEADER));

    /* create the info header */
    mybmp.bmi.biSize = (ulong)sizeof(BITMAPINFOHEADER);
    mybmp.bmi.biWidth  = (ulong)pixels;
    mybmp.bmi.biHeight = (ulong)rasters;
    mybmp.bmi.biPlanes = 1;
    mybmp.bmi.biBitCount = 24;
    mybmp.bmi.biCompression = (ulong) BI_RGB;

    /* BMP scanlines are padded to a multiple of 4 bytes (DWORD) */
    outpacket = (ushort)mybmp.bmi.biWidth * 3;
    while (outpacket%4 != 0)outpacket++;
    mybmp.bmi.biSizeImage = (ulong)outpacket;
	mybmp.bmi.biSizeImage *= mybmp.bmi.biHeight;

    /* create the file header */
    mybmp.bfi.bfType[0] = 'B';
    mybmp.bfi.bfType[1] = 'M';
    mybmp.bfi.bfOffBits = (ulong) sizeof(BMPHEADER);
    mybmp.bfi.bfSize = mybmp.bmi.biSizeImage + mybmp.bfi.bfOffBits;

 	/* write the header for the output BMP */
    c = fwrite((char *)&mybmp.bfi.bfType[0],sizeof(BMPHEADER),1,fp);

    if (c!= 1)outpacket = 0;

return outpacket;
}

void DiffuseError(ushort outpacket)
{
	/*
		http://en.wikipedia.org/wiki/Error_diffusion

		x,y axis (two dimensional) color error diffusion

		in the case of a BMP this disperses the color from the bottom left diagonally through the image
		in the case of some other image format that stores rasters from the top instead of the bottom this would
		disperse the color diagonally from the top left.

        the following scaling ratio is applied:

        1/2 of the gun value of the current pixel is summed with previous pixels as follows:

        1/4 of the gun value of the previous pixel is added to 1/2 the value of the current pixel
        1/8 of the gun value from the previous pixel on the previous line is added to the 1/2 the value of the current pixel
		1/8 of the gun value from the same pixel on the previous line is added to 1/2 the value of the current pixel

	*/

	uchar r2, g2, b2, r4, g4, b4;
    ushort r, g, b;
    sshort i;

	/* create previous pixels for current pixels */
	/* previous pixel on same line */
	b2 = dibscanline1[0];
	g2 = dibscanline1[1];
	r2 = dibscanline1[2];

	/* previous pixel from previous line */
	b4 = dibscanline2[0];
	g4 = dibscanline2[1];
	r4 = dibscanline2[2];


	for (i=0; i < outpacket; i+=3) {

        /* RGB Triples */
	    b = (ushort) dibscanline1[i];
	    g = (ushort) dibscanline1[i+1];
	    r = (ushort) dibscanline1[i+2];

        /* add pixels to create 7/8 ratio of required value */
        /* 4 - current pixels
           2 - previous pixels
           1 - previous pixel below */
	    b *= 4; b += b2; b += b2; b += b4;
	    g *= 4; g += g2; g += g2; g += g4;
	    r *= 4; r += r2; r += r2; r += r4;

        /* add 1 current pixel below - 1/8 ratio of required value */
        /* carry forward b4,g4 and r4 to next pixel */
        /* current pixel below becomes previous pixel below */
        b4 = dibscanline2[i];   b += b4; while (b % 8 != 0) b++; b /=8;
		g4 = dibscanline2[i+1]; g += g4; while (g % 8 != 0) g++; g /=8;
		r4 = dibscanline2[i+2]; r += r4; while (r % 8 != 0) r++; r /=8;

     	/* assign new color to current pixel */
        /* and carry forward b2,g2 and r2 to next pixel */
        /* current pixel becomes previous pixel */

		dibscanline1[i]   = b2 = (uchar)b;
		dibscanline1[i+1] = g2 = (uchar)g;
        dibscanline1[i+2] = r2 = (uchar)r;
	}
}


/* create an error-diffused copy of the input file
   and use that instead */
FILE *ReadDIBFile(FILE *fp, ushort packet)
{
	FILE *fpdib;
	ushort y,outpacket;


    if((fpdib=fopen(dibfile,"wb"))==NULL) {
		printf("Error Opening %s for writing!\n",dibfile);
		return fp;
	}

    outpacket = WriteDIBHeader(fpdib,bmpwidth,bmpheight);
    if (outpacket != packet) {
		fclose(fpdib);
		remove(dibfile);
		printf("Error writing header to %s!\n",dibfile);
		return fp;
	}

    /* seek past extraneous info in header if any */
	fseek(fp,bfi.bfOffBits,SEEK_SET);
	for (y=0;y<bmpheight;y++) {
		fread((char *)&bmpscanline[0],1,packet,fp);
		memcpy(&dibscanline1[0],&bmpscanline[0],packet);
		if (y==0) memcpy(&dibscanline2[0],&bmpscanline[0],packet);
        DiffuseError(packet);
		/* save a copy of the previous line */
		if (diffuse == 2) {
			/* if diffusion is by original value use pure line */
			memcpy(&dibscanline2[0],&bmpscanline[0],packet);
		}
		else {
			/* otherwise use diffused line */
			memcpy(&dibscanline2[0],&dibscanline1[0],packet);
		}
        fwrite((char *)&dibscanline1[0],1,packet,fpdib);

	}
    fclose(fpdib);
    fclose(fp);

    if((fp=fopen(dibfile,"rb"))==NULL) {
		printf("Error Opening %s for reading!\n",dibfile);
   		if((fp=fopen(bmpfile,"rb"))==NULL) {
			printf("Error Opening %s for reading!\n",bmpfile);
			return fp;
		}
	}
    /* read the header stuff into the appropriate structures */
    fread((char *)&bfi.bfType[0],
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);
    return fp;
}


/* helper functions for horizontal resizing */
int ExpandBMPLine(uchar *src, uchar *dest, ushort srcwidth, ushort scale)
{
	int i,j,k;
	unsigned char r,g,b;

	srcwidth *=3;
	for (i=0,j=0,k=0;i<srcwidth;) {
		b = src[i++];
		g = src[i++];
		r = src[i++];

		for (j=0;j<scale;j++) {
			dest[k] = b; k++;
			dest[k] = g; k++;
			dest[k] = r; k++;
		}
	}
	return k;
}

int ShrinkBMPLine(uchar *src, uchar *dest, int srcwidth)
{
	int i,j,k;
	ushort r,g,b;


    scale = srcwidth / 140;

	srcwidth *=3;
	for (i=0,j=0,k=0;k<srcwidth;) {
        b = g = r = 0;
		for (j=0;j<scale;j++) {
			b += src[k++];
			g += src[k++];
			r += src[k++];
		}

		dest[i] = (b/scale);i++;
		dest[i] = (g/scale);i++;
		dest[i] = (r/scale);i++;

	}
	return i;
}


/* shrink 640 or 320 to 140 */
/* uses hgrbuf as a work buffer and output buffer */
void ShrinkPixels(FILE *fp)
{

	int packet = (bmpwidth * 3);

	while (packet%4 != 0)packet++;

	fread((char *)&bmpscanline[0],1,packet,fp);
  	ExpandBMPLine((uchar *)&bmpscanline[0],(uchar *)&dhrbuf[0],bmpwidth,7);
	ShrinkBMPLine((uchar *)&dhrbuf[0],(uchar *)&dhrbuf[0],(ushort)(bmpwidth * 7));
}

/* table-driven scaling from 25 to 24 lines */
void ShrinkLines25to24(FILE *fp, FILE *fp2)
{
	ushort pixel,x,i;
	ushort x1,x2;

	ShrinkPixels(fp);
	memcpy(&dibscanline1[0],&dhrbuf[0],420);
	if (bmpheight == 400) {
		ShrinkPixels(fp);
		for (x=0;x<420;x++) {
			pixel = (ushort) dhrbuf[x];
			pixel += dibscanline1[x];
			dibscanline1[x] = (uchar) (pixel/2);
		}
	}
	for (i=0;i<24;i++) {
		ShrinkPixels(fp);
		memcpy(&dibscanline2[0],&dhrbuf[0],420);
		if (bmpheight == 400) {
			ShrinkPixels(fp);
			for (x=0;x<420;x++) {
				pixel = (ushort) dhrbuf[x];
				pixel += dibscanline2[x];
				dibscanline2[x] = (uchar) (pixel/2);
			}
		}

        for (x=0;x<420;x++) {
			x1 = (ushort)dibscanline1[x];
			x2 = (ushort)dibscanline2[x];
			pixel = (ushort) (x1 * mix25to24[i][0]) + (x2 * mix25to24[i][1]);
			bmpscanline[x] = (uchar)(pixel/25);
		}
	   fwrite((char *)&bmpscanline[0],1,420,fp2);
	   if (i<23)memcpy(&dibscanline1[0],&dibscanline2[0],420);
	}
}


/* 640 x 480 scaled to 140 x 192 */
void ShrinkLines640x480(FILE *fp, FILE *fp2)
{
	ushort pixel1,pixel2,x,i;

	ShrinkPixels(fp);
	memcpy(&dibscanline1[0],&dhrbuf[0],420);
	ShrinkPixels(fp);
	memcpy(&dibscanline2[0],&dhrbuf[0],420);
	ShrinkPixels(fp);
	memcpy(&dibscanline3[0],&dhrbuf[0],420);
	ShrinkPixels(fp);
	memcpy(&dibscanline4[0],&dhrbuf[0],420);
	ShrinkPixels(fp);

 	for (x=0;x<420;x++) {
		pixel1 = (ushort) dibscanline1[x];
		pixel1 += dibscanline2[x];
		pixel1 *= 2;
		pixel1 += dibscanline3[x];
		dibscanline1[x] = (uchar) (pixel1/5);

		pixel2 = (ushort) dhrbuf[x];
		pixel2 += dibscanline4[x];
		pixel2 *= 2;
		pixel2 += dibscanline3[x];
		dibscanline2[x] = (uchar) (pixel2/5);
	}

	fwrite((char *)&dibscanline1[0],1,420,fp2);
	fwrite((char *)&dibscanline2[0],1,420,fp2);
}


void ShrinkLines560x384(FILE *fp, FILE *fp2)
{

	ushort x, pixel, packet = (bmpwidth * 3);

	while (packet%4 != 0)packet++;

	fread((char *)&bmpscanline[0],1,packet,fp);
	ShrinkBMPLine((uchar *)&bmpscanline[0],(uchar *)&dibscanline1[0],bmpwidth);
	fread((char *)&bmpscanline[0],1,packet,fp);
	ShrinkBMPLine((uchar *)&bmpscanline[0],(uchar *)&dibscanline2[0],bmpwidth);
	for (x=0;x<420;x++) {
		pixel = (ushort)dibscanline1[x];
		pixel+= dibscanline2[x];
		bmpscanline[x] = (uchar)(pixel/2);
	}
	fwrite((char *)&bmpscanline[0],1,420,fp2);
}


/* create a resized copy of the input file
   and use that instead */
FILE *ResizeBMP(FILE *fp, sshort resize)
{
	FILE *fp2;
	ushort x,y,packet,outpacket,chunks;
    ushort i,j,r,g,b;
    ulong offset=0L;

#ifdef TURBOC
	if (resize == 0)return NULL;
#endif

    if((fp2=fopen(scaledfile,"wb"))==NULL) {
		printf("Error Opening %s for writing!\n",scaledfile);
		return fp;
	}

    if (justify == 1) outpacket = WriteDIBHeader(fp2,280,192);
	else outpacket = WriteDIBHeader(fp2,140,192);
    if (outpacket != 420 && outpacket != 840) {
		fclose(fp2);
		remove(scaledfile);
		printf("Error writing header to %s!\n",scaledfile);
		return fp;
	}

    packet = bmpwidth * 3;

    if (justify == 1) {

       switch (bmpwidth) {
		   case 640:
		             if (jxoffset > -1) {
                        if (jxoffset > 80) jxoffset = 80;
                        offset+= jxoffset * 3;
					 }
				     else {
		             	offset += 120;
					 }
		             if (bmpheight == 480) {
						 if (jyoffset > -1) {
							if (jyoffset > 96)jyoffset = 0;
							else jyoffset = 96 - jyoffset;
							offset += (1920L * jyoffset);
						 }
						 else {
						 	offset += (1920L * 48);
						}
					 }
		             if (bmpheight == 400) {
						 if (jyoffset > -1) {
							if (jyoffset > 16)jyoffset = 0;
							else jyoffset = 16 - jyoffset;
							offset += (1920L * jyoffset);
						 }
						 else {
						 	offset += (1920L * 8);
						}
					 }
		             break;

		   case 320:
					 if (jxoffset > -1) {
                        if (jxoffset > 40) jxoffset = 40;
                        offset+= jxoffset * 3;
					 }
				     else {
		                offset += 60;
					 }
					 if (jyoffset > -1) {
						if (jyoffset > 8)jyoffset = 0;
						else jyoffset = 8 - jyoffset;
						offset += (960L * jyoffset);
					 }
					 else {
		             	offset += (960L * 4);
					 }
		             break;

	   }

	}

    /* seek past extraneous info in header if any */
	fseek(fp,bfi.bfOffBits+offset,SEEK_SET);

    if (justify == 1) {
		for (y = 0;y< 192;y++) {
		    fread((char *)&dibscanline1[0],1,packet,fp);
		    if (bmpheight == 200) {
				/* no merging at all on 320 x 200 */
				fwrite((char *)&dibscanline1[0],1,outpacket,fp2);
				continue;
			}
			fread((char *)&dibscanline2[0],1,packet,fp);
			for (x = 0,i=0,j=0;x<280;x++) {
				b = (ushort)dibscanline1[i]; b+= dibscanline2[i]; i++;
				g = (ushort)dibscanline1[i]; g+= dibscanline2[i]; i++;
				r = (ushort)dibscanline1[i]; r+= dibscanline2[i]; i++;
				/* half merge (merge vertically) unless merge is turned-on */
				if (merge == 0) {
					i+=3;b*=2;g*=2;r*=2;
				}
				else {
					b += dibscanline1[i]; b+= dibscanline2[i]; i++;
					g += dibscanline1[i]; g+= dibscanline2[i]; i++;
					r += dibscanline1[i]; r+= dibscanline2[i]; i++;
				}
			    bmpscanline[j] = (uchar) (ushort)(b/4);j++;
			    bmpscanline[j] = (uchar) (ushort)(g/4);j++;
			    bmpscanline[j] = (uchar) (ushort)(r/4);j++;
			}
			fwrite((char *)&bmpscanline[0],1,outpacket,fp2);
		}
	}
    else {

    	switch(bmpheight)
    	{
			case 200:
			case 400: chunks = 8;   break;
			case 384: chunks = 192; break;
			case 480: chunks = 96;  break;
		}

		for (y=0;y<chunks;y++) {
       		switch(bmpheight) {
				case 200:
				case 400: ShrinkLines25to24(fp,fp2);break;
				case 480: ShrinkLines640x480(fp,fp2);break;
				case 384: ShrinkLines560x384(fp,fp2);break;
			}
		}
	}
    fclose(fp2);
    fclose(fp);

    if((fp=fopen(scaledfile,"rb"))==NULL) {
		printf("Error Opening %s for reading!\n",scaledfile);
   		if((fp=fopen(bmpfile,"rb"))==NULL) {
			printf("Error Opening %s for reading!\n",bmpfile);
			return fp;
		}
	}
    /* read the header stuff into the appropriate structures */
    fread((char *)&bfi.bfType[0],
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);
    return fp;
}


/* expand monochrome bmp lines to 24-bit bmp lines */
unsigned char msk[]={0x80,0x40,0x20,0x10,0x8,0x4,0x2,0x1};
int reverse = 0;
void ReformatMonoLine()
{
     int i,j,k;
     uchar b = 0, w = 255;

     memcpy(&dibscanline1[0],&bmpscanline[0],70);

     if (reverse == 1) {
		 b = 255;
		 w = 0;
	 }

     for(i=0,j=0;i<70;i++)
     {

        for(k=0;k<8;k++)
        {
            if (dibscanline1[i]&msk[k]) {
				bmpscanline[j] = bmpscanline[j+1] = bmpscanline[j+2] = w;
			}
            else {
				bmpscanline[j] = bmpscanline[j+1] = bmpscanline[j+2] = b;
			}
            j+=3;
        }
     }
}

/* expand 16 color and 256 color bmp lines to 24-bit bmp lines */
void ReformatVGALine()
{
	sshort i, j, packet;
	uchar ch;

    memset(dibscanline1,0,1920);
	if (bmi.biBitCount == 8) {
       memcpy(&dibscanline1[0],&bmpscanline[0],bmpwidth);
    }
    else {
		packet = bmpwidth /2;
		if (bmpwidth%2 != 0) packet++;
		for (i=0,j=0;i<packet;i++) {
			ch = bmpscanline[i] >> 4;
			dibscanline1[j] = ch; j++;
			ch = bmpscanline[i] & 0xf;
			dibscanline1[j] = ch; j++;
		}
	}
	memset(&bmpscanline[0],0,1920);
	for (i=0,j=0;i<bmpwidth;i++) {
		  ch = dibscanline1[i];
	      bmpscanline[j] = sbmp[ch].rgbBlue; j++;
	      bmpscanline[j] = sbmp[ch].rgbGreen; j++;
	      bmpscanline[j] = sbmp[ch].rgbRed; j++;
	 }
}

/* convert 16 color and 256 color bmps to 24 bit bmps */
FILE *ReformatBMP(FILE *fp)
{

    FILE *fp2;
	sshort status = SUCCESS;
	ushort packet, outpacket,y;

	if (bmi.biBitCount == 1) {
		if (bmpwidth != 560 || bmpheight != 192) status = INVALID;
	}
	else {
		if (bmpwidth > 280) {
			status = INVALID;
			switch(bmpwidth) {
				case 640: if (bmpheight == 400 || bmpheight == 480) status = SUCCESS; break;
				case 320: if (bmpheight == 200) status = SUCCESS; break;
				case 560: if (bmpheight == 384) status = SUCCESS; break;
			}
		}
		else {
			if (bmpheight > 192) status = INVALID;
		}
	}

	if (status == INVALID) {
		fclose(fp);
		fp = NULL;
		printf("%s is not a supported size!\n",bmpfile);
		return fp;
	}

	if (bmi.biBitCount == 8)
		fread((char *)&sbmp[0].rgbBlue, sizeof(RGBQUAD)*256,1,fp);
	else if (bmi.biBitCount == 4)
		fread((char *)&sbmp[0].rgbBlue, sizeof(RGBQUAD)*16,1,fp);
	else if (bmi.biBitCount == 1)
	    fread((char *)&sbmp[0].rgbBlue, sizeof(RGBQUAD)*2,1,fp);

    /* seek past extraneous info in header if any */
	fseek(fp,bfi.bfOffBits,SEEK_SET);

    /* align on 4 byte boundaries */
    if (bmi.biBitCount == 1) {
		packet = 72;
	}
    else if (bmi.biBitCount == 8) {
		packet = bmpwidth;
	}
	else {
		packet = bmpwidth / 2;
		if (bmpwidth%2 != 0)packet++;
	}
    while ((packet % 4)!=0)packet++;

    if((fp2=fopen(reformatfile,"wb"))==NULL) {
		printf("Error Opening %s for writing!\n",reformatfile);
		return fp;
	}
    if (bmi.biBitCount == 1) {
		outpacket = WriteDIBHeader(fp2,bmpwidth,bmpheight*2);
	}
	else {
    	outpacket = WriteDIBHeader(fp2,bmpwidth,bmpheight);
	}
    if (outpacket < 1) {
		fclose(fp2);
		remove(reformatfile);
		printf("Error writing header to %s!\n",reformatfile);
		return fp;
	}

  	for (y=0;y<bmpheight;y++) {
		fread((char *)&bmpscanline[0],1,packet,fp);
		if (bmi.biBitCount == 1) ReformatMonoLine();
		else ReformatVGALine();
        fwrite((char *)&bmpscanline[0],1,outpacket,fp2);
        /* double lines for monochrome conversion */
        if (bmi.biBitCount == 1) fwrite((char *)&bmpscanline[0],1,outpacket,fp2);
	}
    fclose(fp2);
    fclose(fp);

    reformat = 1;

    if((fp=fopen(reformatfile,"rb"))==NULL) {
		printf("Error Opening %s for reading!\n",reformatfile);
   		if((fp=fopen(bmpfile,"rb"))==NULL) {
			printf("Error Opening %s for reading!\n",bmpfile);
			return fp;
		}
	}
    /* read the header stuff into the appropriate structures */
    fread((char *)&bfi.bfType[0],
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);
    return fp;
}


/* overlay */
sshort OpenMaskFile()
{

	sshort status = INVALID;
	ushort i, width=0, height=0;
	double dummy;
	int c;

    if (overlay == 0) return status;

    overlay = 0;
    fpmask = fopen(maskfile,"rb");
    if (NULL == fpmask) {
		printf("Error opening %s\n",maskfile);
		return status;
	}

    for (;;) {

		c = fread((char *)&maskbmp.bfi.bfType[0],sizeof(BMPHEADER),1,fpmask);

		if (c!= 1) {
			/* printf("header read returned %d\n",c); */
			break;
		}

    	if (maskbmp.bmi.biCompression==BI_RGB &&
     		maskbmp.bfi.bfType[0] == 'B' && maskbmp.bfi.bfType[1] == 'M' &&
     		maskbmp.bmi.biPlanes==1 && maskbmp.bmi.biBitCount == 8) {
				width = (ushort) maskbmp.bmi.biWidth;
				height = (ushort) maskbmp.bmi.biHeight;
		}

        /* this ensures that only full-screen output is masked */
        /* it doesn't make sense to mix image fragment routines into here right now */
        if (width != 140 || height != 192) {
			/* printf("width = %d, height = %d\n",width,height); */
			break;
		}

        fread((char *)&maskpalette[0].rgbBlue, sizeof(RGBQUAD)*256,1,fpmask);

		/* build a remap array for DHGR colors  */
		for (i=0;i<256;i++) {
			remap[i] = GetMedColor(maskpalette[i].rgbRed,
			                       maskpalette[i].rgbGreen,
			                       maskpalette[i].rgbBlue,&dummy);
		}
		fseek(fpmask,bfi.bfOffBits,SEEK_SET);
		status = SUCCESS;
		overlay = 1;
		break;
	}

    if (status == INVALID){
		/* puts("Failed!"); */
		fclose(fpmask);
		fpmask = NULL;
		if (quietmode == 1)printf("Error loading %s\n",maskfile);
	}
	else {
		if (quietmode == 1)printf("Loaded mask %s\n",maskfile);
	}

    return status;
}

/* 1. reads a 24 bit BMP file in the range from 1 x 1 to 280 x 192 */
/* 2. writes a DHGR screen image or optionally a DHGR image fragment */
/* 3. also creates an optional preview file...
   		when preview is on... also leaves an error-diffused dib file
   		in place if error diffusion is also turned-on */
sshort Convert()
{

    FILE *fp, *fpdib, *fpreview;
    sshort status = INVALID, resize = 0;
	ushort x,x1,x2,y,i,packet, outpacket, width, dwidth, red, green, blue;
	uchar r,g,b,drawcolor;
	ulong pos, prepos;

    /* if using a mask file, open it now */
    /* leave it open throughout the conversion session */
    /* it will be closed in main before exiting */
	if (overlay == 1)OpenMaskFile();

    if((fp=fopen(bmpfile,"rb"))==NULL) {
		printf("Error Opening %s for reading!\n",bmpfile);
		return status;
	}
    /* read the header stuff into the appropriate structures */
    fread((char *)&bfi.bfType[0],
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);

    /* reformat to 24 bit */
    if (bmi.biCompression==BI_RGB &&
        bfi.bfType[0] == 'B' && bfi.bfType[1] == 'M' && bmi.biPlanes==1 &&
       ((bmi.biBitCount == 8) || (bmi.biBitCount == 4))) {

		bmpwidth = (ushort) bmi.biWidth;
		bmpheight = (ushort) bmi.biHeight;

	    fp = ReformatBMP(fp);
	    if (fp == NULL) return INVALID;
	}

    if (bmi.biCompression==BI_RGB &&
        bfi.bfType[0] == 'B' && bfi.bfType[1] == 'M' &&
        bmi.biPlanes==1 && bmi.biBitCount == 24) {

		bmpwidth = (ushort) bmi.biWidth;
		bmpheight = (ushort) bmi.biHeight;

        /* resize some classic screen sizes */
        if (bmpwidth == 320 && bmpheight == 200)
           resize = 1;
        else  if (bmpwidth == 640 && bmpheight == 400)
           resize = 2;
        else if (bmpwidth == 640 && bmpheight == 480)
           resize = 3;
        else if (bmpwidth == 560 && bmpheight == 384)
           resize = 4;

        if (resize != 0) {
    		memset(&bmpscanline[0],0,1920);
    		memset(&dibscanline1[0],0,1920);
    		memset(&dibscanline2[0],0,1920);
    		memset(&dibscanline3[0],0,1920);
			fp = ResizeBMP(fp,resize);
			if (fp == NULL) return INVALID;
			bmpwidth = (ushort) bmi.biWidth;
			bmpheight = (ushort) bmi.biHeight;
		}

		if (scale == 0) {
			if (bmpwidth > 140) scale = 1;
		}

		if (scale == 1) {
			width = bmpwidth;
			dwidth = (bmpwidth+1)/2;
            if (bmpwidth  > 0 && bmpwidth < 281 &&
                bmpheight > 0 && bmpheight < 193) status = SUCCESS;
		}
		else {
			width = bmpwidth * 2;
			dwidth = bmpwidth;
            if (bmpwidth  > 0 && bmpwidth < 141 &&
                bmpheight > 0 && bmpheight < 193) status = SUCCESS;

		}

	}

    if (status == INVALID) {
		fclose(fp);
		printf("%s is in the wrong format!\n",bmpfile);
		return status;
	}


	packet = bmpwidth * 3;
    /* BMP scanlines are padded to a multiple of 4 bytes (DWORD) */
	while ((packet % 4) != 0) packet++;

    /* error diffusion option */
    if (diffuse != 0) {
		/* clear buffers */
    	memset(&bmpscanline[0],0,960);
    	memset(&dibscanline1[0],0,960);
    	memset(&dibscanline2[0],0,960);
		fp = ReadDIBFile(fp, packet);
		if (fp == NULL) return INVALID;
	}

	if (preview!=0) {
		fpreview = fopen(previewfile,"wb+");

		if (fpreview != NULL) {
			outpacket = WriteDIBHeader(fpreview,width,bmpheight);
			if (outpacket == 0) {
				fclose(fpreview);
				remove(previewfile);
				printf("Error writing header to %s!\n",previewfile);
				preview = 0;
			}
			else {
				/* pad the preview file */
				memset(&dibscanline1[0],0,960);
    			for (y=0;y<bmpheight;y++) fwrite((char *)&dibscanline1[0],1,outpacket,fpreview);
    			/* set the seek distance to scanline 0 in the preview file */
    			prepos = (ulong) (bmpheight - 1);
    			prepos *= outpacket;
    			prepos += mybmp.bfi.bfOffBits;
			}

		}
		else {
			printf("Error opening %s for writing!\n",previewfile);
			preview = 0;
		}
	}


	/* read BMP from top scanline to bottom scanline */
    pos = (ulong) (bmpheight - 1);
    pos *= packet;
    pos += bfi.bfOffBits;

    /* clear buffers */
    dhrclear();
	memset(&bmpscanline[0],0,960);
	memset(&previewline[0],0,960);

	if (dither != 0) {
		/* sizeof(sshort) * 320 */
		memset(&redDither[0],0,640);
		memset(&greenDither[0],0,640);
		memset(&blueDither[0],0,640);
		memset(&redSeed[0],0,640);
		memset(&greenSeed[0],0,640);
		memset(&blueSeed[0],0,640);
	}

	for (y=0;y<bmpheight;y++,pos-=packet) {
		fseek(fp,pos,SEEK_SET);
		fread((char *)&bmpscanline[0],1,packet,fp);

        if (overlay == 1)ReadMaskLine(y);

		if (scale == 1) {
			for (x = 0,i = 0, x1=0; x < bmpwidth; x++) {
				/* get even pixel values */
				b = bmpscanline[i]; i++;
				g = bmpscanline[i]; i++;
				r = bmpscanline[i]; i++;
				x++;

                /* get odd pixel values */
               	if (x < bmpwidth) {
					if (merge == 0) {
					  blue  = (ushort)b;
					  green = (ushort)g;
					  red   = (ushort)r;
					  i+=3;
					}
					else {
					  blue  = (ushort)bmpscanline[i]; i++;
					  green = (ushort)bmpscanline[i]; i++;
					  red   = (ushort)bmpscanline[i]; i++;
					}

				}
				else {
					/* if no odd pixel double-plot the last pixel */
                	if (merge == 0) {
						blue  = (ushort)b;
						green = (ushort)g;
						red   = (ushort)r;
					}
					else {
 						/* merge with background color
                   	   	   on some fragments the background color might already be padded-out
                		*/
						blue  = (ushort)rgbArray[backgroundcolor][2];
						green = (ushort)rgbArray[backgroundcolor][1];
						red   = (ushort)rgbArray[backgroundcolor][0];
					}
			  	}

				blue  += b;
				green += g;
				red   += r;

				b = (uchar) (blue/2);
				g = (uchar) (green/2);
				r = (uchar) (red/2);

                if (dither == 0) {

					maskpixel = 0;
					if (overlay == 1) {
						overcolor = maskline[x/2];
						/* clearcolor is the transparent color for the mask */
						/* if the overlay color is some other color then the pixel is overlaid
						   with the mask color */
						if (overcolor != clearcolor) maskpixel = 1;
					}
					if (maskpixel == 1) {
						drawcolor = (uchar)overcolor;
					}
					else {
						/* get nearest color index from currently selected conversion palette */
						drawcolor = GetDrawColor(r,g,b,x/2,y);
					}

					/* plot to DHGR buffer */
					dhrplot(x/2,y,drawcolor);
					if (preview == 1) {
						/* plot preview using currently selected preview palette */
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][BLUE]; x1++;
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][GREEN]; x1++;
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][RED]; x1+=4;
					}
				}
				else {
					/* Floyd-Steinberg Etc. dithering */
					/* values are already seeded from previous line(s) */
					x2 = x/2;

					AdjustShortPixel(1,(sshort *)&redDither[x2],(sshort)r);
					AdjustShortPixel(1,(sshort *)&greenDither[x2],(sshort)g);
					AdjustShortPixel(1,(sshort *)&blueDither[x2],(sshort)b);
				}
			}
		}
		else {
			/* merge has no meaning unless we are scaling */
			for (x = 0,i = 0,x1=0; x < bmpwidth; x++) {
				b = bmpscanline[i]; i++;
				g = bmpscanline[i]; i++;
				r = bmpscanline[i]; i++;

				if (dither != 0) {
					/* Floyd-Steinberg Etc. dithering */
					/* values are already seeded from previous line(s) */
					AdjustShortPixel(1,(sshort *)&redDither[x],(sshort)r);
					AdjustShortPixel(1,(sshort *)&greenDither[x],(sshort)g);
					AdjustShortPixel(1,(sshort *)&blueDither[x],(sshort)b);
				}
				else {
					maskpixel = 0;
					if (overlay == 1) {
						overcolor = maskline[x];
						/* clearcolor is the transparent color for the mask */
						/* if the overlay color is some other color then the pixel is overlaid
						   with the mask color */
						if (overcolor != clearcolor) maskpixel = 1;
					}
					if (maskpixel == 1) {
						drawcolor = (uchar)overcolor;
					}
					else {
						/* get nearest color index from currently selected conversion palette */
                		drawcolor = GetDrawColor(r,g,b,x,y);
					}
					/* plot to DHGR buffer */
					dhrplot(x,y,drawcolor);
					if (preview == 1) {
						/* plot preview using currently selected preview palette */
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][BLUE]; x1++;
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][GREEN]; x1++;
						previewline[x1] = previewline[x1+3] = rgbPreview[drawcolor][RED]; x1+=4;
					}
				}
			}
		}

        if (dither != 0) {
		   /* Floyd-Steinberg dithering */
		   FloydSteinberg(y,dwidth);
		   /* seed next line - promote nearest forward array to
		      current line */
		   memcpy(&redDither[0],&redSeed[0],640);
		   memcpy(&greenDither[0],&greenSeed[0],640);
		   memcpy(&blueDither[0],&blueSeed[0],640);

           /* seed first seed - promote furthest forward array
              to nearest forward array */
		   memcpy(&redSeed[0],&redSeed2[0],640);
		   memcpy(&greenSeed[0],&greenSeed2[0],640);
		   memcpy(&blueSeed[0],&blueSeed2[0],640);

		   /* clear last seed - furthest forward array */
 		   /* this is not used in all the error diffusion dithers */
 		   /* - but dithers like atkinson use 2 foward arrays */
 		   /* - in dithers that use only one forward array this does no harm */
 		   /* somewhat brute force but simple code */
 		   memset(&redSeed2[0],0,640);
		   memset(&greenSeed2[0],0,640);
		   memset(&blueSeed2[0],0,640);
		}

		if (preview != 0) {
			/* write the preview line to the preview file */
			fseek(fpreview,prepos,SEEK_SET);
			fwrite((char *)&previewline[0],1,outpacket,fpreview);
			prepos -= outpacket;
		}

	}

	fclose(fp);

	if (preview != 0) {
		fclose(fpreview);
		if (quietmode != 0) printf("Preview file %s created!\n",previewfile);
	}

    if (debug == 0) {
		if (diffuse  != 0) remove(dibfile);
		if (resize != 0) remove(scaledfile);
		if (reformat != 0) remove(reformatfile);
	}

    if (savedhr() != SUCCESS) return INVALID;
    if (savesprite() != SUCCESS) return INVALID;

	return SUCCESS;

}

sshort ConvertMono()
{

    FILE *fp, *fpreview;
    sshort status = INVALID;
	ushort x,y,i,packet, outpacket, red, green, blue, verbatim;
	ulong pos, prepos;

    if((fp=fopen(bmpfile,"rb"))==NULL) {
		printf("Error Opening %s for reading!\n",bmpfile);
		return status;
	}
    /* read the header stuff into the appropriate structures */
    fread((char *)&bfi.bfType[0],
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);

	bmpwidth = (ushort) bmi.biWidth;
	bmpheight = (ushort) bmi.biHeight;

    /* monochrome verbatim conversion */
	if (bmpwidth == 560 && bmpheight == 192 && bmi.biBitCount == 1) {
		verbatim = 1;
	}
	/* color to dithered monochrome conversion */
	else if (bmpwidth == 560 && bmpheight == 384 && bmi.biBitCount != 1) {
		verbatim = 2;
	}
	else {
		fclose(fp);
		puts("Invalid size for Monochrome conversion!");
		return status;
	}

    /* reformat to 24 bit */
    if (bmi.biCompression==BI_RGB &&
        bfi.bfType[0] == 'B' && bfi.bfType[1] == 'M' && bmi.biPlanes==1 &&
       ((bmi.biBitCount == 8) || (bmi.biBitCount == 4) || (bmi.biBitCount == 1))) {
	    fp = ReformatBMP(fp);
	    if (fp == NULL) return INVALID;
	}

    if (bmi.biCompression==BI_RGB &&
        bfi.bfType[0] == 'B' && bfi.bfType[1] == 'M' &&
        bmi.biPlanes==1 && bmi.biBitCount == 24) {

		bmpwidth = (ushort) bmi.biWidth;
		bmpheight = (ushort) bmi.biHeight;

		status = SUCCESS;
	}

    if (status == INVALID) {
		fclose(fp);
		printf("%s is in the wrong format!\n",bmpfile);
		return status;
	}


	packet = bmpwidth * 3;
    /* BMP scanlines are padded to a multiple of 4 bytes (DWORD) */
	while ((packet % 4) != 0) packet++;

	if (preview!=0) {
		fpreview = fopen(previewfile,"wb+");

		if (fpreview != NULL) {
			outpacket = WriteDIBHeader(fpreview,bmpwidth,bmpheight);
			if (outpacket == 0) {
				fclose(fpreview);
				remove(previewfile);
				printf("Error writing header to %s!\n",previewfile);
				preview = 0;
			}
			else {
				/* pad the preview file */
				memset(&dibscanline1[0],0,960);
    			for (y=0;y<bmpheight;y++) fwrite((char *)&dibscanline1[0],1,outpacket,fpreview);
    			/* set the seek distance to scanline 0 in the preview file */
    			prepos = (ulong) (bmpheight - 1);
    			prepos *= outpacket;
    			prepos += mybmp.bfi.bfOffBits;
			}

		}
		else {
			printf("Error opening %s for writing!\n",previewfile);
			preview = 0;
		}
	}


	/* read BMP from top scanline to bottom scanline */
    pos = (ulong) (bmpheight - 1);
    pos *= packet;
    pos += bfi.bfOffBits;

    /* clear buffers */
    memset(dhrbuf,0,16384);
	memset(&bmpscanline[0],0,1920);
	memset(&previewline[0],0,1920);

	/* sizeof(sshort) * 640 */
	memset(&redDither[0],0,1280);
	memset(&greenDither[0],0,1280);
	memset(&blueDither[0],0,1280);
	memset(&redSeed[0],0,1280);
	memset(&greenSeed[0],0,1280);
	memset(&blueSeed[0],0,1280);

	for (y=0;y<192;y++,pos-=packet) {
		fseek(fp,pos,SEEK_SET);
		fread((char *)&bmpscanline[0],1,packet,fp);
		pos-=packet;
		fread((char *)&bmpscanline2[0],1,packet,fp);

		for (x = 0,i = 0; x < bmpwidth; x++, i+=3) {

			blue = (ushort)bmpscanline[i];
			green = (ushort)bmpscanline[i+1];
			red = (ushort)bmpscanline[i+2];

			if (verbatim == 2) {
				blue  +=  bmpscanline[i];
				green += bmpscanline[i+1];
				red   += bmpscanline[i+2];
			}
			/* Floyd-Steinberg Etc. dithering */
			/* values are already seeded from previous line(s) */
			AdjustShortPixel(1,(sshort *)&redDither[x],(sshort)red/verbatim);
			AdjustShortPixel(1,(sshort *)&greenDither[x],(sshort)green/verbatim);
			AdjustShortPixel(1,(sshort *)&blueDither[x],(sshort)blue/verbatim);
		}

	   /* Floyd-Steinberg dithering */
	   FloydSteinberg(y,bmpwidth);
	   /* seed next line - promote nearest forward array to
		  current line */
	   memcpy(&redDither[0],&redSeed[0],1280);
	   memcpy(&greenDither[0],&greenSeed[0],1280);
	   memcpy(&blueDither[0],&blueSeed[0],1280);

	   /* seed first seed - promote furthest forward array
		  to nearest forward array */
	   memcpy(&redSeed[0],&redSeed2[0],1280);
	   memcpy(&greenSeed[0],&greenSeed2[0],1280);
	   memcpy(&blueSeed[0],&blueSeed2[0],1280);

	   /* clear last seed - furthest forward array */
	   /* this is not used in all the error diffusion dithers */
	   /* - but dithers like atkinson use 2 forward arrays */
	   /* - in dithers that use only one forward array this does no harm */
	   /* somewhat brute force but simple code */
	   memset(&redSeed2[0],0,1280);
	   memset(&greenSeed2[0],0,1280);
	   memset(&blueSeed2[0],0,1280);


		if (preview != 0) {
			/* write the preview line to the preview file */
			fseek(fpreview,prepos,SEEK_SET);
			fwrite((char *)&previewline[0],1,outpacket,fpreview);
			prepos -= outpacket;
			fseek(fpreview,prepos,SEEK_SET);
			fwrite((char *)&previewline[0],1,outpacket,fpreview);
			prepos -= outpacket;
		}

	}

	fclose(fp);

	if (preview != 0) {
		fclose(fpreview);
		if (quietmode != 0) printf("Preview file %s created!\n",previewfile);
	}

    if (debug == 0) {
		if (reformat != 0) remove(reformatfile);
	}

    if (savedhr() != SUCCESS) return INVALID;
	return SUCCESS;

}



char *title = "Bmp2DHR (c) Copyright Bill Buckels 2014. All Rights Reserved.";

char *usage[] = {
"Usage:      \"b2d input.bmp options\"",
"BMP format: 16 color, 256 color, or 24 bit Version 3 uncompressed",
"Output:     DHGR Color Full Screen (default) or Sprite (optional)",
"Free Scaled Input Sizes: Sprite Test Mode or Sprite Output (option F)",
"  Full Scale: from 1 x 1 to 140 x 192 (default)",
"  Half Scale: from 2 x 1 to 280 x 192 (scaling option S2)",
"Fixed Scale Input Sizes: Full Screen Output (default)",
"  140 x 192 - Full Scale (prescaled BMPs from 256 color gifs, etc)",
"  280 x 192 - Half Scale (double x) (CiderPress and Appewin output, etc)",
"  320 x 200 - legacy graphics, etc",
"  560 x 384 - screen grabs, even scale input (use 640 x 480 for photos)",
"  640 x 400 - legacy graphics (640 x 480 is better for photos etc)",
"  640 x 480 - photos and legacy graphics, wikipedia images, wallpaper, etc",
"Quick Options: (so you can remember how to use Bmp2DHR)",
"  \"b2d input.bmp photo\"  - turns-on default dithering",
"  \"b2d input.bmp art\"    - turns-on default cross-hatching",
"  \"b2d input.bmp both\"   - turns-on combined dithering-hatching",
"  \"b2d input.bmp sprite\" - turns-on default sprite output",
"  \"b2d input.bmp bin\"    - changes output to AUX,BIN from A2FC",
"  \"b2d help\" (or b2d h)  - displays detailed usage.",
"Type \"b2d h\" for more information or download the users manual:",
"  http://www.appleoldies.ca/cc65/docs/dhgr/bmp2dhr.pdf",
NULL};


char *help[] = {
	"Usage:   \"b2d input.bmp options\"",
	"Input:   prescaled BMP - 140 x 192 or 280 x 192 nominal resolution",
	"Range:   default range          - from 1 x 1 to 140 x 192",
	"         with s2 scaling option - from 2 x 1 to 280 x 192",
	"Output:  full screen DHGR A2FC (default) or AUX,BIN (optional) files",
	"         Optional image fragment (sprite) output",
	"Classic Input: prescaled BMP (16 color, 256 color, or 24-bit)",
	"         320x 200, 560 x 384, 640 x 400, 640 x 480",
	"Options:(see User's Manual for additional options and quick commands)",
	"a - output aux,bin - instead of a2fc default",
	"b - background color 1-15 (0 by default)",
	"d - dither (d1-d9) - default d1 (Floyd-Steinberg)",
	"f - image fragment output over-ride (off by default)",
	"e - error diffusion - e2 (default) or e4 (optional)",
	"m - horizontal merge - s2 mode only",
	"p - (p0,p1,p2,p3,p4,p5) use built-in palette.",
	"    Wikipedia Apple II color palette (p4) is the default.",
	"q - disable quietmode (can be redirected to file).",
	"    creates an embedded sprite array when option f is on.",
    "s2 - horizontal half-scale (s1 full-scale by default)",
    "t - turn-on CiderPress file attribute preservation tags.",
	"v - output preview bmp (v0,v1,v2,v3,v4,v5 - palette colors)",
    "x - (x1,x3) pattern for 2 x 2 cross-hatching (x2 is default)",
	"y - (y1,y3) color distance adjustment (y2 is normal)",
	"z - (z1 - z50) threshold setting for cross-hatching",
	NULL};


void phelp(char category)
{

	sshort i;
	char ch;

    puts(title);
	ch = toupper(category);

	switch (ch) {

		case 'H':
		default: for (i=0;help[i] != NULL;i++) puts(help[i]);
	}
}

void pusage(void)
{
	sshort i;

    puts(title);
	for (i=0;usage[i] != NULL;i++) puts(usage[i]);
}



/* ------------------------------------------------------------------------ */
/* palette reader and helper functions                                      */
/* adapted from Clipshop                                                    */
/* ------------------------------------------------------------------------ */

/* strip line feeds from ascii file lines... */

void nocr(char *ptr) {
  int idx;
  for (idx = 0; ptr[idx] != 0; idx++)
    if (ptr[idx] == LFEED || ptr[idx] == CRETURN || ptr[idx] == '#')
      ptr[idx] = 0;
}

/*
squeeze redundant whitespace from lines read-in from a palette file
(leave only a single space character)
this is important if the user has created their own palette file
by hand... since they may accidentally type more than one whitespace
between RGB values...

Also, phototsyler version 2 palette file lines are fixed width,
right justified so we need to massage these for our reader...
*/
void SqueezeLine(char *ptr)
{
  int idx, jdx, len;
  char buf[128];

  idx = 0;
  while (ptr[idx] == ' ')idx++;  /* remove leading whitespace */
  strcpy(buf, &ptr[idx]);

  jdx = 0;
  ptr[jdx] = ASCIIZ;

  for (idx = 0; buf[idx] != ASCIIZ; idx++) {
    if (buf[idx] == 9) buf[idx] = ' ';         /* no tabs please */
    if (buf[idx] == ',') buf[idx] = ' ';       /* no commas please */
    if (buf[idx] == ' ' && buf[idx +1] == ' ')
      continue;
    /* truncate if any non-numeric characters */
    if ((buf[idx] < '0' || buf[idx] > '9') && buf[idx] != ' ')
      buf[idx] = ASCIIZ;
    ptr[jdx] = buf[idx]; jdx++;
    ptr[jdx] = ASCIIZ;
  }

  /* remove trailing whitespace...
    this occurrs during parsing of photostyler */
  len = strlen(ptr);
  while (len > 0) {
    len--;
    if (ptr[len] != ' ')
      break;
    ptr[len] = ASCIIZ;
  }
}

/* split the RGB triple from a text line read-in from an
   ascii palette file. */
int ReadPaletteLine(unsigned char *ptr, unsigned char *palptr, unsigned int colordepth)
{
  int red, green, blue, idx, spaces = 0;

  red = atoi(ptr);
  if (red < 0 || red > 255) return INVALID;

  /* there must be at least 3 fields */
  for (idx = 0; ptr[idx] != 0; idx++) {
    if (ptr[idx] == ' ' && ptr[idx+1] >= '0' && ptr[idx+1] <= '9') {
       spaces++;
       switch(spaces) {
         case 1:
           green = atoi(&ptr[idx+1]);
           if (green < 0 || green > 255) return INVALID;
           break;
         case 2:
           blue = atoi(&ptr[idx+1]);
           if (blue < 0 || blue > 255) return INVALID;
           break;
       }
    }
  }

  if (spaces<2)
    return INVALID;

  if (colordepth == 6) {
     palptr[0] = (uchar)red << 2;
     palptr[1] = (uchar)green << 2;
     palptr[2] = (uchar)blue << 2;
   }
   else {
     palptr[0] = (uchar)red;
     palptr[1] = (uchar)green;
     palptr[2] = (uchar)blue;
   }
   return SUCCESS;

}

/* check version if Paintshop palette since JASC may change someday */
/* also check Aldus version although that product is old... */

/* The Top Half of NeoPaint Windows Palettes are the same as their */
/* DOS palettes so we use the 6 bit color values and handle both   */
/* file types the same way... so no worry about neopaint versions. */

char *Gimp = "GIMP Palette"; /* followed by RGB values and comments */

/* NeoPaint and PaintShop Pro headers
   3 lines followed by RGB values */
char *NeoPaint  = "NeoPaint Palette File";
char *PaintShop = "JASC-PAL";
char *PaintShopVersion = "0100";

/* Aldus photostyler
   3 lines followed by RGB values */
char *AldusPal = "CWPAL";
char *AldusClr = "CWCLR";           /* partial palettes */
char *AldusVersion = "100";

#define GENERIC 1
#define GIMP 2
#define JASC 3
#define NEO 4
#define ALDUS 5

sshort GetUserPalette(char *name)
{
	FILE *fp;
	char buf[128];
	int cnt=16;
	sshort status = INVALID;
	unsigned colordepth=8,paltype=GENERIC;

	fp = fopen(name,"r");
	if (fp == NULL) return status;

	for (;;) {
  		if (NULL == fgets(buf, 128, fp)) {
    		fclose(fp);
    		break;
		}
  		nocr(buf);
  		SqueezeLine(buf);

        /* check for some known palette types */
  		if (strcmp(Gimp, buf)==0) paltype = GIMP;
  		else if (strcmp(PaintShop, buf)==0) paltype = JASC;
  		else if (strcmp(NeoPaint, buf)==0) {
  			colordepth = 6;
    		paltype = NEO;
  		}
  		else if (strcmp(AldusPal, buf) == 0 || strcmp(AldusClr, buf) == 0) {
	  		paltype = ALDUS;
  		}
  		/* if not a known type then assume it's just a simple csv */

  		status = SUCCESS;
		switch(paltype)
		{
			case GENERIC: rewind(fp); break;

			case JASC:
			case NEO:
			case ALDUS:
			    /* check 2 remaining header lines */
				status = INVALID;
				if (NULL == fgets(buf, 128, fp)) break;
				nocr(buf);
				SqueezeLine(buf);
				if (paltype == JASC && strcmp(PaintShopVersion, buf)!=0)break;
				if (paltype == ALDUS && strcmp(AldusVersion, buf) != 0)break;
				if (NULL == fgets(buf, 128, fp)) break;
				cnt = atoi(buf);
				if (cnt < 16) break;
				status = SUCCESS;
		}
		if (status == INVALID) break;

        memset(&rgbUser[0][0],0,48);
        cnt = 0;
		while (fgets(buf,128,fp) != NULL) {
			if (buf[0] == '#') continue;
			if (strlen(buf) < 5) continue;
  			nocr(buf);
  			SqueezeLine(buf);
    		if (INVALID == ReadPaletteLine(buf,(uchar *)&rgbUser[cnt][0],colordepth)) continue;
    		cnt++;
    		if (cnt > 15)break;
		}
		break;
	}
	fclose(fp);

	if (cnt < 15) {
		printf("%s contains only %d colors!",name,cnt);
	}
	if (status == INVALID) {
		printf("%s is not a valid palette file!",name);
	}
	return status;
}



/* returns the Apple II Hires drawcolor 0-15 */
/* a double hi-res pixel can occur at any one of 7 positions */
/* in a 4 byte block which spans aux and main screen memory */
/* the horizontal resolution is 140 pixels */
int dhrgetpixel(int x,int y)
{
    int xoff, pattern, idx;
    unsigned char *ptraux, *ptrmain,c1, c2, d1, d2;

    pattern = (x%7);
	xoff = HB[y] + ((x/7) * 2);
    ptraux  = (unsigned char *) &dhrbuf[xoff-0x2000];
    ptrmain = (unsigned char *) &dhrbuf[xoff];


	switch(pattern)
	{
		/* left this here for reference

		unsigned char dhrpattern[7][4] = {
		0,0,0,0,
		0,0,0,1,
		1,1,1,1,
		1,1,2,2,
		2,2,2,2,
		2,3,3,3,
        3,3,3,3};
        */

        /* compare colors in the input file to color patterns and return drawcolor */
        /* somewhat inelegant but lazy to read and debug if a problem */
		case 0: c1 = ptraux[0] &0x0f;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][0] & 0x0f;
				  if (d1 == c1) return idx;
				}
		        break;
		case 1: c1 = ptraux[0] & 0x70;
		        c2 = ptrmain[0] & 0x01;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][0] & 0x70;
				  d2 = dhrbytes[idx][1] & 0x01;
				  if (d1 == c1 && d2 == c2) return idx;
				}
		        break;
		case 2: c1 = ptrmain[0] & 0x1e;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][1] & 0x1e;
				  if (d1 == c1) return idx;
				}
		        break;
		case 3: c1 = ptrmain[0] & 0x60;
		        c2 = ptraux[1] & 0x03;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][1] & 0x60;
				  d2 = dhrbytes[idx][2] & 0x03;
				  if (d1 == c1 && d2 == c2) return idx;
				}
                break;
		case 4: c1 = ptraux[1] & 0x3c;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][2] & 0x3c;
				  if (d1 == c1) return idx;
				}
		        break;
		case 5: c1 = ptraux[1] & 0x40;
 		        c2 = ptrmain[1] & 0x07;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][2] & 0x40;
				  d2 = dhrbytes[idx][3] & 0x07;
				  if (d1 == c1 && d2 == c2) return idx;
				}
                break;
		case 6: c1 = ptrmain[1] & 0x78;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][3] & 0x78;
				  if (d1 == c1) return idx;
				}
 		        break;
	}

    return INVALID;

}


uchar BMP_header[] ={
0x42, 0x4D, 0x36, 0x76, 0x02, 0x00, 0x00, 0x00,
0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x18, 0x01, 0x00, 0x00, 0xC0, 0x00,
0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x76, 0x02, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

int save_to_bmp24(void)
{

    FILE *fp;
    uchar tempr, tempg, tempb;
	int x,y,y2,idx;

	fp = fopen(previewfile,"wb");
	if (NULL == fp)return INVALID;

/* write header for 280 x 192 x 24 bit bmp */
    fwrite(BMP_header,1,sizeof(BMP_header),fp);


/* write rgb triples and double each pixel to preserve the apsect ratio */

    y2 = 191;
   	for (y = 0; y< 192; y++) {

	   for (x = 0; x < 140; x++) {
		  idx = dhrgetpixel(x,y2);

          /* range check */
          if (idx < 0 || idx > 15)idx = 0; /* default black */

		  tempr = rgbPreview[idx][0];
		  tempg = rgbPreview[idx][1];
		  tempb = rgbPreview[idx][2];

		  /* reverse order */
		  fputc(tempb, fp);
		  fputc(tempg, fp);
		  fputc(tempr, fp);

		  /* double-up */
		  fputc(tempb, fp);
		  fputc(tempg, fp);
		  fputc(tempr, fp);
	   }
	   y2 -= 1;
    }

    fclose(fp);
    return SUCCESS;

}

/* experimental only */
/* 4 x 6 font titling */
/* 140 / 4 = 35 characters across */
/* 192 / 6 = 32 lines down */
sshort GetUserTextFile()
{
	FILE *fp;
	char buf[128];
	int x,y,i,cnt=0;

	fp = fopen(usertextfile,"r");
	if (NULL == fp) return INVALID;

	for (i=0,y=0;i<32;i++,y+=6) {
  		if (NULL == fgets(buf, 128, fp)) {
    		break;
		}
  		nocr(buf);
  		if (buf[0] == 0) continue;
  	    buf[35] = 0;
  	    for (x=0;x<3;x++) {
			thumbDHGR(buf,x,y,0,255,'L');
			thumbDHGR(buf,x,y+1,0,255,'L');
			thumbDHGR(buf,x,y+2,0,255,'L');
		}
  		if (backgroundcolor == 0)thumbDHGR(buf,1,y+1,15,255,'L');
  		else thumbDHGR(buf,1,y+1,(unsigned char)backgroundcolor,255,'L');
  		cnt++;
	}
	fclose(fp);

	if (cnt!= 0 && preview != 0) save_to_bmp24();

	return SUCCESS;
}


int cmpstr(char *str, char *cmp)
{
	int i;

	if (strlen(cmp) != strlen(str)) return INVALID;
	for (i=0;str[i] != 0;i++) {
		if (toupper(cmp[i]) != toupper(str[i])) return INVALID;
	}
	return SUCCESS;
}

int main(int argc, char **argv)
{
	sshort idx,jdx,kdx,palidx = 4,previewidx = 4,tags=0,status;
	uchar c, ch, *wordptr;


    if (argc < 2) {
		pusage();
		return (1);
	}

	if (argc == 2) {
		/* check for help first */
    	wordptr = (uchar *)&argv[1][0];
		ch = toupper(wordptr[0]);
		if (ch == '-') {
	    	wordptr = (uchar *)&argv[1][1];
		}
		ch = toupper(wordptr[0]);
		if (ch == '?') ch = 'h';
		if (cmpstr(wordptr,"help") == SUCCESS) {
			phelp(0);
			return (1);
		}
		else if (ch == 'h' && strlen(wordptr) < 3) {
			c = toupper(wordptr[1]);
			phelp(c);
			return (1);
		}
	}

    usertextfile[0] = 0;

    /* getopts */
    if (argc > 2) {
    	for (idx = 2; idx < argc; idx++) {
			/* switch character is optional */
	   		wordptr = (uchar *)&argv[idx][0];
	   		ch = toupper(wordptr[0]);
	   		if (ch == '-') {
				wordptr = (uchar *)&argv[idx][1];
                ch = toupper(wordptr[0]);
			}

			if (cmpstr(wordptr,"debug") == SUCCESS) {
				debug = 1;
				continue;
			}

			/* so-called "quick" commands */
			if (cmpstr(wordptr,"photo") == SUCCESS) {
				dither = FLOYDSTEINBERG;
				continue;
			}
			if (cmpstr(wordptr,"art") == SUCCESS) {
				threshold = 25;
				xmatrix = 2;
				continue;
			}
			if (cmpstr(wordptr,"both") == SUCCESS) {
				dither = FLOYDSTEINBERG;
				threshold = 15;
				xmatrix = 2;
				continue;
			}
			if (cmpstr(wordptr,"sprite") == SUCCESS) {
				outputtype = SPRITE_OUTPUT;
				continue;
			}

			if (cmpstr(wordptr,"BIN") == SUCCESS) {
				applesoft = 1;
				continue;
			}

			if (cmpstr(wordptr,"sum") == SUCCESS) {
				errorsum = 1;
				continue;
			}

			if (cmpstr(wordptr,"mono") == SUCCESS || cmpstr(wordptr,"reverse") == SUCCESS) {
				mono = 1;
				if (dither == 0) dither = FLOYDSTEINBERG;
				if (cmpstr(wordptr,"reverse") == SUCCESS) reverse = 1;
				continue;
			}


		    switch(ch) {
				case 'A': /* output AUX,BIN - default is A2FC */
				          applesoft = 1;
				          break;
				case 'B': /* background color 1-15 (0 by default) */
				          c = toupper(wordptr[1]);
				          /* non numeric argument so check for color names */
				          /* black is not an option */
				          if (c > 57) {
				            if (cmpstr("red", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 1;
				          	else if (cmpstr("dark blue", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 2;
				          	else if (cmpstr("purple", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 3;
				          	else if (cmpstr("dark green", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 4;
				          	else if (cmpstr("gray", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 5;
				          	else if (cmpstr("medium blue", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 6;
				          	else if (cmpstr("light blue", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 7;
				          	else if (cmpstr("brown", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 8;
				          	else if (cmpstr("orange", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 9;
				          	else if (cmpstr("grey", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 10;
				          	else if (cmpstr("pink", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 11;
				          	else if (cmpstr("light green", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 12;
				          	else if (cmpstr("yellow", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 13;
				          	else if (cmpstr("aqua", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 14;
				          	else if (cmpstr("white", (char *)&wordptr[1]) == SUCCESS)backgroundcolor = 15;
				          	else backgroundcolor = 0;
							break;
						  }

				          jdx = atoi((char *)&wordptr[1]);
				          if (jdx > 0 && jdx < 16) backgroundcolor = (uchar)jdx;
				          break;

				case 'C': globalclip = 1;
				          break;

                case 'D': dither = FLOYDSTEINBERG;

                          if (ReadCustomDither((char *)&wordptr[1]) == SUCCESS) {
							  break;
						  }

						  ch = toupper(wordptr[1]);
						  if (ch == 'X') {
							  wordptr++;
							  serpentine = 1;
						  }

                		  jdx = atoi((char *)&wordptr[1]);
                		  if (jdx > 0 && jdx < 10) dither = jdx;
                		  else {
							  ch = toupper(wordptr[1]);
							  switch(ch) {
								  case 'F': dither = FLOYDSTEINBERG;break;
								  case 'J': dither = JARVIS;break;
								  case 'S': dither = STUCKI;
								            ch = toupper(wordptr[2]);
								            if (ch == 'I') dither = SIERRA;
								            else if (ch == '2') dither = SIERRATWO;
								            else if (ch == 'L') dither = SIERRALITE;
								            break;
								  case 'A': dither = ATKINSON;break;
								  case 'B': dither = BURKES;
								            ch = toupper(wordptr[2]);
								            if (ch == 'B') dither = 9;
								            break;
							  }
						  }
                          break;


				case 'E':
                          /* error diffusion default = E2 */
                          diffuse = 2;
                          jdx = atoi((char *)&wordptr[1]);
                          /* E4 */
                          if (jdx == 4) diffuse = 4;
                          break;

				case 'F': /* image fragment - off by default */
				          outputtype = SPRITE_OUTPUT; break;

				case 'J':
						   /* scaling of larger sizes is pixel by pixel
						      when justification is selected */
						   justify = 1;
				           ch = toupper(wordptr[1]); /* justify */
						   switch(ch) {
							   case 'L': jxoffset = atoi((char *)&wordptr[2]);
							             break;
							   case 'T': jyoffset = atoi((char *)&wordptr[2]);
							             break;

						   }
						   break;

                case 'M': /* M2 - horizontal merge - S2 mode only */
                          /* by default every second pixel is skipped */
                          merge = 1;
                          break;

                case 'O': /* use an 8 bit overlay file - must be 140 x 192 */
                          /* transparent color must be set or default is 128,128,128 - color 5 */
                          if (wordptr[1] == 0) break;
 				          c = toupper(wordptr[1]);
 				          /* non numeric argument so check for color names */
 				          if (c > 57) {
							c = 16;
							if (cmpstr("black", (char *)&wordptr[1]) == SUCCESS)c = 0;
 				            else if (cmpstr("red", (char *)&wordptr[1]) == SUCCESS)c = 1;
 				          	else if (cmpstr("dark blue", (char *)&wordptr[1]) == SUCCESS)c = 2;
 				          	else if (cmpstr("purple", (char *)&wordptr[1]) == SUCCESS)c = 3;
 				          	else if (cmpstr("dark green", (char *)&wordptr[1]) == SUCCESS)c = 4;
 				          	else if (cmpstr("gray", (char *)&wordptr[1]) == SUCCESS)c = 5;
 				          	else if (cmpstr("medium blue", (char *)&wordptr[1]) == SUCCESS)c = 6;
 				          	else if (cmpstr("light blue", (char *)&wordptr[1]) == SUCCESS)c = 7;
 				          	else if (cmpstr("brown", (char *)&wordptr[1]) == SUCCESS)c = 8;
 				          	else if (cmpstr("orange", (char *)&wordptr[1]) == SUCCESS)c = 9;
 				          	else if (cmpstr("grey", (char *)&wordptr[1]) == SUCCESS)c = 10;
 				          	else if (cmpstr("pink", (char *)&wordptr[1]) == SUCCESS)c = 11;
 				          	else if (cmpstr("light green", (char *)&wordptr[1]) == SUCCESS)c = 12;
 				          	else if (cmpstr("yellow", (char *)&wordptr[1]) == SUCCESS)c = 13;
 				          	else if (cmpstr("aqua", (char *)&wordptr[1]) == SUCCESS)c = 14;
 				          	else if (cmpstr("white", (char *)&wordptr[1]) == SUCCESS)c = 15;
 				          	if (c!= 16) {
 				        		clearcolor = c; break;
							}
						  }
                          if (wordptr[1] == '0' && wordptr[2] == 0) {
							  clearcolor = 0; break;
						  }
						  jdx = atoi((char *)&wordptr[1]);
						  if (jdx > 0 && jdx < 16) {
							  if (wordptr[2] == 0) {
								  clearcolor = jdx; break;
							  }
							  if (wordptr[3] == 0) {
							  	  clearcolor = jdx; break;
							  }
						  }
                          overlay = 1;
                		  strcpy(maskfile,(char *)&wordptr[1]);
                		  /* maskfile must have an extension or .bmp is assumed */
                		  /* this avoids typing extensions which I dislike doing */
                		  /* so I enforce extensions. any questions? */
                	      kdx = 999;
                	      for (jdx=0;maskfile[jdx]!=0;jdx++) {
							  if (maskfile[jdx] == '.')kdx = jdx;
						  }
						  if (kdx == 999)strcat(maskfile,".bmp");
                		  break;

				case 'V': /* create preview file */
				          preview = 1;
				          if (wordptr[1] == 0) break;
				          if (cmpstr(wordptr,"vbmp") == SUCCESS) {
							  preview = vbmp = 1;
							  break;
						  }

				case 'P': /* palette settings */
				          c = toupper(wordptr[1]);
 				          /* check for palette names */
 				          if (c > 57) {
							c = 255;
							if (cmpstr("kegs32", (char *)&wordptr[1]) == SUCCESS)c = 0;
							else if (cmpstr("kegs", (char *)&wordptr[1]) == SUCCESS)c = 0;
 				            else if (cmpstr("ciderpress", (char *)&wordptr[1]) == SUCCESS)c = 1;
 				          	else if (cmpstr("old applewin", (char *)&wordptr[1]) == SUCCESS)c = 2;
 				          	else if (cmpstr("new applewin", (char *)&wordptr[1]) == SUCCESS)c = 3;
 				          	else if (cmpstr("applewin", (char *)&wordptr[1]) == SUCCESS)c = 3;
 				          	else if (cmpstr("wikipedia", (char *)&wordptr[1]) == SUCCESS)c = 4;
 				          	else if (cmpstr("wiki", (char *)&wordptr[1]) == SUCCESS)c = 4;
 				          	else if (cmpstr("rgb", (char *)&wordptr[1]) == SUCCESS) {
								/* second grey level */
								c = 4;
								wikipedia[10][0] = wikipedia[10][1] = wikipedia[10][2] = 192;
							}
							else if ((cmpstr("hgro", (char *)&wordptr[1]) == SUCCESS) ||
							         (cmpstr("hgrg", (char *)&wordptr[1]) == SUCCESS)) {
								c = 4;
								/* Low-resolution colors
								   0 (black),
								   3 (purple),
								   6 (medium blue),
								   9 (orange),
								   12 (light green) and
								   15 (white) were also available in high-resolution mode */
								wikipedia[1][0] = wikipedia[1][1] = wikipedia[1][2] = 0;
								wikipedia[2][0] = wikipedia[2][1] = wikipedia[2][2] = 0;
								wikipedia[4][0] = wikipedia[4][1] = wikipedia[4][2] = 0;
								wikipedia[5][0] = wikipedia[5][1] = wikipedia[5][2] = 0;
								wikipedia[7][0] = wikipedia[7][1] = wikipedia[7][2] = 0;
								wikipedia[8][0] = wikipedia[8][1] = wikipedia[8][2] = 0;
								wikipedia[10][0] = wikipedia[10][1] = wikipedia[10][2] = 0;
								wikipedia[11][0] = wikipedia[11][1] = wikipedia[11][2] = 0;
								wikipedia[13][0] = wikipedia[13][1] = wikipedia[13][2] = 0;
								wikipedia[14][0] = wikipedia[14][1] = wikipedia[14][2] = 0;

								if (cmpstr("hgro", (char *)&wordptr[1]) == SUCCESS) {
									wikipedia[3][0] = wikipedia[3][1] = wikipedia[3][2] = 0;
									wikipedia[12][0] = wikipedia[12][1] = wikipedia[12][2] = 0;

								}
								else {
									wikipedia[6][0] = wikipedia[6][1] = wikipedia[6][2] = 0;
									wikipedia[9][0] = wikipedia[9][1] = wikipedia[9][2] = 0;
								}
							}
 				          	else if (cmpstr("sheldon simms", (char *)&wordptr[1]) == SUCCESS)c = 5;
 				          	else if (cmpstr("todhr", (char *)&wordptr[1]) == SUCCESS)c = 5;
 				          	else if (cmpstr("canvas", (char *)&wordptr[1]) == SUCCESS)c = 7;
 				          	else if (cmpstr("bmp", (char *)&wordptr[1]) == SUCCESS)c = 8;
 				          	else if (cmpstr("win16", (char *)&wordptr[1]) == SUCCESS)c = 8;
 				          	else if (cmpstr("xmp", (char *)&wordptr[1]) == SUCCESS)c = 9;
 				          	else if (cmpstr("win32", (char *)&wordptr[1]) == SUCCESS)c = 9;
 				          	else if (cmpstr("vga", (char *)&wordptr[1]) == SUCCESS)c = 10;
 				          	else if (cmpstr("pcx", (char *)&wordptr[1]) == SUCCESS)c = 11;
 				          	if (c!= 255) {
								if (ch == 'P') palidx = c;
							 	else previewidx = c;
 				        		break;
							}
						  }

				          jdx = GetUserPalette((char *)&wordptr[1]);
				          if (jdx == SUCCESS) {
							 if (ch == 'P') palidx = 6;
							 else previewidx = 6;
						  }
						  else {

				          	c = toupper(wordptr[1]);
				          	if (c == 'R') {
								jdx = 4;
								wikipedia[10][0] = wikipedia[10][1] = wikipedia[10][2] = 192;
							}
				          	else if (c == 'K' || c == 'C' || c == 'O' || c == 'N' || c == 'W' || c == 'S') {
							  jdx=0;
							  switch(c) {
								  case 'S': jdx++;
								  case 'W': jdx++;
								  case 'N': jdx++;
								  case 'O': jdx++;
								  case 'C': jdx++;
							  }

						  	}
						  	else {
							 	if (c < 48 || c > 59) break;
				          	 	jdx = atoi((char *)&wordptr[1]);
						  	}
						  	/* palettes 7-11 are undocumented legacy palettes */
						  	/* palette 6 is a user palette file */
						  	/* palettes 0-5 are the documented palettes */
				          	if (jdx > -1 && jdx < 12) {
							  	if (ch == 'P') palidx = jdx;
							  	else previewidx = jdx;
						  	}
						  }
				          break;
				case 'Q': quietmode = 0;
				          break;

				case 'R': /* reduced or increased color bleed
				             by percentage (for dithering only) */

						  jdx = atoi((char *)&wordptr[1]);
						  if ((jdx > 0 && jdx < 101) || (jdx < 0 && jdx > -101)) colorbleed = 100 + jdx;
						  break;

				case 'S': /* default scale - S1 full scale */
				          jdx = atoi((char *)&wordptr[1]);
				          /* S2 - double scaled */
				          if (jdx == 2) scale = 1;
				          break;
				case 'T': /* use ciderpress tags - off by default */
				          tags = 1;
				          break;

                case 'X': /* pattern setting for general purpose 2 x 2 cross-hatching */
                          xmatrix = 2;
                          if (threshold == 0) threshold = 25;
                          /* optional pattern setting for general purpose 2 x 2 cross-hatching */
                          jdx = atoi((char *)&wordptr[1]);
                          if (jdx == 1 || jdx == 3) xmatrix = jdx;
                          break;
                case 'Y': /* increase or decrease color - non-cross-hatched ouput */
                          /* this can eventually be replaced by a saturation adjustment or
                             a hue correction or something else */
                          ymatrix = 1;
                          jdx = atoi((char *)&wordptr[1]);
                          if (jdx == 2 || jdx == 3) ymatrix = jdx;
                          break;

				case 'Z': /* threshold setting for general purpose 2 x 2 cross-hatching */
				          /* and for brightening and darkening of colors */
				          threshold = 25;
				          if (xmatrix == 0)xmatrix = 2;
				          jdx = atoi((char *)&wordptr[1]);
				          /* allow up to 50% adjustment on RGB values */
				          /* surely that's enough */
				          if (jdx > 0 && jdx < 51) threshold = jdx;
				          break;

			}
		}
	}

    /* embedding of image fragments only */
    if (outputtype != SPRITE_OUTPUT) quietmode = 1;

    jdx = 999;
	strcpy(fname, argv[1]);
	for (idx = 0; fname[idx] != (uchar)0; idx++) {
		if (fname[idx] == '.') {
			jdx = idx;
		}
	}
    if (jdx != 999) fname[jdx] = (uchar)0;

    sprintf(bmpfile,"%s.bmp",fname);
    sprintf(dibfile,"%s.dib",fname);
#ifdef MSDOS
	tags = 0;
    sprintf(previewfile,"%s.pmp",fname);
    sprintf(scaledfile,"%s.smp",fname);
    sprintf(reformatfile,"%s.rmp",fname);
    sprintf(vbmpfile,"%s.vmp",fname);
#else
    sprintf(previewfile,"%s_Preview.bmp",fname);
    sprintf(scaledfile,"%s_Scaled.bmp",fname);
    sprintf(reformatfile,"%s_Reformat.bmp",fname);
    sprintf(vbmpfile,"%s_VBMP.bmp",fname);
#endif
    /* user titling file */
    sprintf(usertextfile,"%s.txt",fname);

    /* upper case basename for Apple II Output */
    for (idx = 0; fname[idx] != (uchar)0; idx++) {
		ch = toupper(fname[idx]);
		fname[idx] = ch;
	}

    /* CiderPress File Attribute Preservation Tags */
    if (tags == 1) {
		sprintf(mainfile,"%s.BIN#062000",fname);
		sprintf(auxfile,"%s.AUX#062000",fname);
		sprintf(a2fcfile,"%s.A2FC#062000",fname);
		sprintf(spritefile,"%s.DHR#062000",fname);
	}
    else {
		/* tags are off by default */
		/* unadorned file names */
		sprintf(mainfile,"%s.BIN",fname);
		sprintf(auxfile,"%s.AUX",fname);
#ifdef MSDOS
		sprintf(a2fcfile,"%s.2FC",fname);
#else
		sprintf(a2fcfile,"%s.A2FC",fname);
#endif
		sprintf(spritefile,"%s.DHR",fname);
	}

	if (mono == 1) {
		palidx = previewidx = 4;
		memset(&wikipedia[0][0],0,45);
	}

	GetBuiltinPalette(palidx,previewidx);
    InitDoubleArrays();

    if (mono == 1) status = ConvertMono();
    else status = Convert();

    /* close mask file if any before exiting */
    if (NULL != fpmask) fclose(fpmask);
    if (status == INVALID) return (1);

	return SUCCESS;
}


